<?php

namespace Netzperfekt\QRCodeGenerator;

use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\Label\Font\OpenSans;
use Endroid\QrCode\Label\Label;
use Endroid\QrCode\Logo\Logo;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\RoundBlockSizeMode;
use Netzperfekt\QRCodeGenerator\Dto\QRCodeContent;
use Netzperfekt\QRCodeGenerator\Dto\QRCodeGenerated;
use Netzperfekt\QRCodeGenerator\Dto\QRCodeOptions;
use Netzperfekt\QRCodeGenerator\Enums\QRCodeErrorCorrection;

class QRCodeGenerator implements QRCodeGeneratorInterface
{
    public function generate(QRCodeContent $payload, QRCodeOptions $options): QRCodeGenerated
    {
        // https://github.com/endroid/qr-code

        $errorCorrectionLevel = match($options->errorCorrection) {
            QRCodeErrorCorrection::Low => ErrorCorrectionLevel::Low,
            QRCodeErrorCorrection::Medium => ErrorCorrectionLevel::Medium,
            QRCodeErrorCorrection::Quartile => ErrorCorrectionLevel::Quartile,
            QRCodeErrorCorrection::High => ErrorCorrectionLevel::High,
        };

        $data = $payload->uri;
        if(empty($data))
        {
            return QRCodeGenerated::from();
        }

        $colorValuesForeground = colorHexStringToBytes($options->colorForeground);
        $colorValuesBackground = colorHexStringToBytes($options->colorBackground);
        $colorValuesText = colorHexStringToBytes($options->colorText);

        $qrCode = QrCode::create($data)
            ->setEncoding(new Encoding('UTF-8'))
            ->setErrorCorrectionLevel($errorCorrectionLevel)
            ->setSize(300)
            ->setMargin(10)
            ->setRoundBlockSizeMode(RoundBlockSizeMode::Margin)
            ->setForegroundColor(
                new Color($colorValuesForeground[0], $colorValuesForeground[1], $colorValuesForeground[2])
            )
            ->setBackgroundColor(
                new Color($colorValuesBackground[0], $colorValuesBackground[1], $colorValuesBackground[2])
            );

        // TODO optional logo einbauen
        // TODO ggf. noch text größe und fontname variabel?
        // https://github.com/endroid/qr-code

        $label = Label::create($payload->ctaText)
            ->setFont(new OpenSans(25))
                      ->setTextColor(new Color($colorValuesText[0], $colorValuesText[1], $colorValuesText[2]));

        /* TODO aus options auslesen
        $logo = Logo::create(getcwd() . '/laravel.svg')
            ->setResizeToHeight(100)
            ->setResizeToWidth(100)
            ->setPunchoutBackground(true);
        */

        $writer = new SvgWriter();
        $result = $writer->write($qrCode, label: $label, logo: $logo ?? null);

        return QRCodeGenerated::from([
            'uri'        => $payload->uri,
            'dataUri'    => $result->getDataUri(),
            'dataString' => $result->getString(),
            'payload'    => $data,
            'title'      => $payload->title,
            'ctaText'    => $payload->ctaText
        ]);
    }
}
