<?php

namespace Netzperfekt\SaasBase\Filament\Pages;

use Filament\Actions\Action;
use Filament\Forms\Components\Tabs;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Resources\Components\Tab;
use Filament\Support\Exceptions\Halt;
use Illuminate\Support\Facades\Auth;
use Netzperfekt\SaasBase\Events\GetSettingsFieldsEvent;
use Netzperfekt\SaasBase\Models\Settings as SettingsModel;

class Settings extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-cog';
    protected static string $view = 'saas-base::filament.pages.settings';
    protected static ?int $navigationSort = 99;

    public ?array $data = [];

    protected function getActions(): array
    {
        return [];
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')->keyBindings(['command+s', 'ctrl+s'])
                ->label(__('save'))
                ->submit('data')
        ];
    }

    protected function getFormStatePath(): string
    {
        return 'data';
    }

    protected function getFormSchema(): array
    {
        $settingsFields = self::collectSettingsFields();

        $tabs = [];
        $n = 0;

        foreach($settingsFields as $settingsField)
        {
            $tabs[] = Tabs\Tab::make('settings-tab' . $n)
                ->label($settingsField['label'])
                ->schema($settingsField['fields']);

            $n++;
        }

        return [
            Tabs::make('settings-tabs')
                ->tabs($tabs)
                ->persistTabInQueryString()
        ];
    }

    public function mount(): void
    {
        $settings = SettingsModel::all();

        $formValues = [];
        foreach($settings as $setting)
        {
            $formValues[$setting->key] = json_decode($setting->value, true);
        }

        $this->form->fill();
        $defaults = $this->form->getState();

        $this->form->fill($formValues + $defaults);
    }

    public function submit(): void
    {
        try
        {
            foreach ($this->form->getState() as $key => $value)
            {
                SettingsModel::updateOrCreate(
                    ['key' => $key],
                    ['value' => json_encode($value)]
                );

                activity()
                    ->causedBy(Auth::user())
                    ->withProperties(['key' => $key, 'value' => $value])
                    ->log('settings');
            }
            $this->callHook('afterSave');
        }
        catch (Halt $exception)
        {
            return;
        }

        Notification::make()
            ->title(__('saved'))
            ->success()
            ->send();
    }

    public static function getNavigationGroup(): ?string
    {
        return __(config('saas-base.settings_group'));
    }

    public static function getNavigationLabel(): string
    {
        return __(config('saas-base.settings_label'));
    }

    public function getTitle(): string
    {
        return __(config("saas-base.settings_label"));
    }

    public static function collectSettingsFields()
    {
        $settingsFields = array_filter(event(new GetSettingsFieldsEvent()));

        usort($settingsFields, function($a, $b)
        {
            if(isset($a['sort']) && isset($b['sort'])) {
                return $a['sort'] ?? 0 > $b['sort'] ?? 0;
            }

            return strcmp($a['label'], $b['label']);
        });

        return $settingsFields;
    }
}
