<?php

namespace Netzperfekt\SaasBase\Filament\Resources\UserResource;

use Filament\Facades\Filament;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\ToggleButtons;
use Filament\Forms\Components\ViewField;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\IconPosition;
use Filament\Support\Enums\IconSize;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasBase\Filament\Resources\UserResource\Pages\CreateUser;
use Netzperfekt\SaasBase\Filament\Resources\UserResource\Pages\EditUser;
use Netzperfekt\SaasBase\Filament\Resources\UserResource\Pages\ListUsers;
use Netzperfekt\SaasBase\Models\User;
use Netzperfekt\SaasBase\Models\UserType;
use Netzperfekt\SaasBase\SaasBase;
use STS\FilamentImpersonate\Impersonate;
use Tapp\FilamentTimezoneField\Forms\Components\TimezoneSelect;

class UserResource extends Resource
{
    protected static ?string $model = User::class;
    protected static ?string $recordTitleAttribute = 'name';
    protected static bool $isScopedToTenant = false;

    protected static ?string $navigationIcon = 'heroicon-o-user-circle';
    protected static ?int $navigationSort = 3;

    public static function getNavigationGroup(): ?string
    {
        return __('user-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::count();
    }

    public static function getModelLabel(): string
    {
        return __('user');
    }

    public static function getPluralModelLabel(): string
    {
        return __('user');
    }

    public static function scopeEloquentQueryToTenant(Builder $query, ?Model $tenant): Builder
    {
        return $query;
    }

    public static function canCreate(): bool
    {
        return auth()->user()->isSuperAdmin();
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(3)
            ->schema([
                Section::make('Benutzerdaten')
                    ->label(__('user-data'))
                    ->columnSpan(2)
                    ->columns(3)
                    ->schema([
                        TextInput::make('name')
                            ->label(__('name'))
                            ->required(),

                        TextInput::make('email')
                            ->label(__('email'))
                            ->email()
                            ->unique(ignoreRecord: true)
                            ->required(),

                        ViewField::make('avatar')
                            ->view(SaasBase::PACKAGE_NAME . '::filament.avatar')
                            ->formatStateUsing(fn ($record) => $record ? Filament::getUserAvatarUrl($record) : '')
                            ->label('')
                            ->dehydrated(false),

                        Select::make('usertype_id')
                            ->label(__('audience'))
                            ->options(UserType::active()->pluck('title', 'id'))
                            ->default(UserType::getDefault()->id)
                            ->required(),
                    ]),

                Section::make(__('options'))
                    ->columnSpan(1)
                    ->schema([
                        Toggle::make('is_active')
                            ->label(__('active'))
                            ->onColor('success')
                            ->offColor('danger')
                            ->default(true)
                            ->disabled(fn($record) => $record?->isSuperAdmin()),

                        Toggle::make('superadmin')
                            ->label(__('super-admin')),

                        Toggle::make('newsletter')
                            ->label(__('newsletter')),

                        TimezoneSelect::make('timezone')
                            ->label(__('timezone'))
                            ->default(config('saas-base.user_default_timezone'))
                            ->searchable()
                            ->required(),

                        Placeholder::make('api_tokens')
                            ->visible(fn($record) => count($record?->tokens ?? []) > 0)
                            ->content(function ($record)
                            {
                                $s = 'Active Tokens: <b>';
                                $s .= count($record?->tokens ?? []);
                                $s .= '</ul>';

                                return new HtmlString($s);
                            })
                    ])
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('avatar')
                    ->circular()
                    ->getStateUsing(fn ($record) => Filament::getUserAvatarUrl($record))
                    ->label(''),

                Tables\Columns\TextColumn::make('name')
                    ->label(__('name'))
                    ->tooltip(fn ($record) => $record->isSuperAdmin() ? __('super-admin') : '')
                    ->icon(fn ($record) => $record->isSuperAdmin() ? 'fas-crown' : '')
                    ->iconColor('warning')
                    ->iconPosition(IconPosition::After)

                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('email')
                    ->label(__('email'))
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('teams.name')
                    ->label(__('user-team'))
                    ->badge()
                    ->color(Color::Orange)
                    ->sortable(),

                Tables\Columns\TextColumn::make('userType.title')
                    ->label(__('audience'))
                    ->badge()
                    ->sortable(),

                Tables\Columns\ToggleColumn::make('newsletter')
                    ->label(__('newsletter'))
                    ->disabled()
                    ->sortable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->label(__('registered'))
                    ->dateTime('d.m.y')
                    ->sortable(),

                Tables\Columns\TextColumn::make('last_login')
                    ->label(__('last-login'))
                    ->dateTime('d.m.y / H:i')
                    ->sortable(),

                Tables\Columns\ToggleColumn::make('is_active')
                    ->label(__('active'))
                    ->disabled()
                    ->onColor('success')
                    ->offColor('danger')
                    ->sortable()
            ])

            ->defaultSort('superadmin', 'desc')

            ->filters([
                Tables\Filters\Filter::make('superadmin')
                    ->label(__('super-admin'))
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('superadmin', true)),

                Tables\Filters\SelectFilter::make('team')
                    ->label(__('user-team'))
                    ->relationship('teams', 'name')
                    ->searchable()
                    ->preload()
            ])

            ->actions([
                Impersonate::make()->redirectTo(route('filament.app.tenant')),

                Tables\Actions\EditAction::make()
                    ->label('')
                    ->tooltip(__('edit'))
                    ->iconSize(IconSize::Medium),

                Tables\Actions\DeleteAction::make()
                    ->label('')
                    ->tooltip(__('Delete'))
                    ->iconSize(IconSize::Medium)
            ])
            ->bulkActions([
            ])
        ;
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListUsers::route('/'),
            'create' => CreateUser::route('/create'),
            'edit' => EditUser::route('/{record}/edit'),
        ];
    }
}
