<?php

namespace Netzperfekt\SaasBase\Models;

use Filament\Models\Contracts\FilamentUser;
use Filament\Models\Contracts\HasTenants;
use Filament\Panel;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

class User extends Authenticatable implements FilamentUser, HasTenants, MustVerifyEmail
{
    use HasApiTokens, HasFactory, Notifiable;
    use LogsActivity;

    protected $fillable = [
        'is_active',
        'superadmin',
        'name',
        'email',
        'password',
        'timezone',
        'config',
        'newsletter',
        'usertype_id',
        'email_verified_at',
        'last_login'
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'last_login' => 'datetime',
        'password' => 'hashed',
        'config' => 'array'
    ];

    public function canAccessPanel(Panel $panel): bool
    {
        if($panel->getId() == 'admin')
        {
            return $this->isSuperAdmin();
        }

        return true;
    }

    public function canImpersonate()
    {
        return $this->isSuperAdmin();
    }

    public function getTenants(Panel $panel): Collection
    {
        return $this->teams;
    }

    public function teams(): BelongsToMany
    {
        return $this->belongsToMany(Team::class);
    }

    public function mainTeam(): Team
    {
        return $this->teams()->first();
    }

    // TODO eigentlich besser in _src/Domain aufgehoben
    public function userType(): BelongsTo
    {
        return $this->belongsTo(UserType::class, 'usertype_id');
    }

    public function canAccessTenant(Model $tenant): bool
    {
        return $this->teams->contains($tenant);
    }

    public function isActive(): bool
    {
        return $this->is_active;
    }

    public function isSuperAdmin(): bool
    {
        return $this->superadmin ?? false;
    }

    public function getCurrentTeamId(): int
    {
        return $this->current_team_id;
    }

    public function setCurrentTeamId(int $teamId): void
    {
        $this->current_team_id = $teamId;
    }

    public function scopeSuperAdmin(Builder $query): void
    {
        $query->where('superadmin', true);
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['name', 'email', 'is_active', 'newsletter', 'usertype_id'])
            ->dontLogIfAttributesChangedOnly(['last_login', 'updated_at']);
    }

    public function getTokens(string $name): Collection
    {
        return $this->tokens()->where('name', 'LIKE', $name . '%')->get();
    }

    public static function getConfig(string $key, $default = null): mixed
    {
        return Auth::user()->config[$key] ?? $default;
    }
}
