<?php

namespace Netzperfekt\SaasBase;

use Filament\Contracts\Plugin;
use Filament\Facades\Filament;
use Filament\Forms\Components\DateTimePicker;
use Filament\Http\Middleware\Authenticate;
use Filament\Http\Middleware\DisableBladeIconComponents;
use Filament\Http\Middleware\DispatchServingFilamentEvent;
use Filament\Navigation\MenuItem;
use Filament\Navigation\NavigationItem;
use Filament\Panel;
use Filament\Support\Facades\FilamentView;
use Filament\Tables\Columns\TextColumn;
use Filament\View\PanelsRenderHook;
use Guava\FilamentKnowledgeBase\KnowledgeBasePlugin;
use Illuminate\Cookie\Middleware\AddQueuedCookiesToResponse;
use Illuminate\Cookie\Middleware\EncryptCookies;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Foundation\Http\Middleware\VerifyCsrfToken;
use Illuminate\Routing\Middleware\SubstituteBindings;
use Illuminate\Session\Middleware\AuthenticateSession;
use Illuminate\Session\Middleware\StartSession;
use Illuminate\View\Middleware\ShareErrorsFromSession;
use Illuminate\View\View;
use Netzperfekt\SaasBase\Filament\Admin\AdminDashboard;
use Netzperfekt\SaasBase\Filament\Admin\AppDashboard;
use Netzperfekt\SaasBase\Filament\Classes\GravatarsProvider;
use Netzperfekt\SaasBase\Filament\Pages\Auth\EditProfile;
use Netzperfekt\SaasBase\Filament\Pages\Auth\Register;
use Netzperfekt\SaasBase\Filament\Pages\Changelog;
use Netzperfekt\SaasBase\Filament\Pages\Tenancy\EditTeamProfile;
use Netzperfekt\SaasBase\Filament\Pages\Tenancy\RegisterTeam;
use Netzperfekt\SaasBase\Middleware\ExcludeFromRobots;
use RalphJSmit\Filament\MediaLibrary\Media\Models\MediaLibraryFolder;
use RalphJSmit\Filament\MediaLibrary\Media\Models\MediaLibraryItem;

class SaasBase implements Plugin
{
    public final const PACKAGE_NAME = 'saas-base';

    protected bool $isEnabled = false;
    protected bool $canRegister = false;
    protected bool $isZendeskWidgetEnabled = false;
    protected bool $isHelpEnabled = false;
    protected bool $isChangelogEnabled = false;

    public static function make(): SaasBase
    {
        return new SaasBase();
    }

    public function getId(): string
    {
        return self::PACKAGE_NAME;
    }

    public function register(Panel $panel): void
    {
        $activePages = [];

        $panel
            ->default()
            ->login()
            ->passwordReset()
            ->emailVerification()
            ->tenantRegistration(RegisterTeam::class)
            ->tenantProfile(EditTeamProfile::class)
            ->profile(EditProfile::class)
            ->defaultAvatarProvider(GravatarsProvider::class)
            ->userMenuItems([
                'profile' => MenuItem::make()->label(fn(): string => __('profile') . ': ' . auth()->user()->name ?? '')
            ])
            ->middleware([
                EncryptCookies::class,
                AddQueuedCookiesToResponse::class,
                StartSession::class,
                AuthenticateSession::class,
                ShareErrorsFromSession::class,
                VerifyCsrfToken::class,
                SubstituteBindings::class,
                DisableBladeIconComponents::class,
                DispatchServingFilamentEvent::class,
                ExcludeFromRobots::class
            ])
            ->authMiddleware([
                Authenticate::class,
            ]);

        if ($panel->getId() == 'admin')
        {
            $activePages[] = AdminDashboard::class;
        }
        elseif ($panel->getId() == 'app')
        {
            $activePages[] = AppDashboard::class;
        }

        if ($this->getIsEnabled()) {
            $panel->discoverResources(
                in: __DIR__ . '/Filament/Resources', for: 'Netzperfekt\\SaasBase\\Filament\\Resources'
            );
        }

        if ($this->canRegister())
        {
            $panel->registration(Register::class);
        }

        if ($this->getHelpEnabled())
        {
            $panel->plugins([
                KnowledgeBasePlugin::make()
                    ->modalPreviews()
                    ->slideOverPreviews()
                    ->disableKnowledgeBasePanelButton()
                    ->helpMenuRenderHook(PanelsRenderHook::GLOBAL_SEARCH_AFTER)
            ]);

            // routes are not resolved at this time - so defer url generation with (fn() => ...)
            $panel->navigationItems([
                NavigationItem::make('Hilfe')
                    ->url(fn() => \Guava\FilamentKnowledgeBase\Facades\KnowledgeBase::url(
                            \Guava\FilamentKnowledgeBase\Facades\KnowledgeBase::panel()
                        ) . '/' . config('filament-knowledge-base.docs-root-url'))
                    ->icon('heroicon-o-question-mark-circle')
                    ->openUrlInNewTab(true)
            ]);
        }

        if ($this->getChangelogEnabled())
        {
            $activePages[] = ChangeLog::class;
        }

        $panel->pages($activePages);
    }

    public function boot(Panel $panel): void
    {
        $this->bootFilament();
    }

    private function bootFilament()
    {
        FilamentView::registerRenderHook(
            PanelsRenderHook::GLOBAL_SEARCH_AFTER,
            fn(): View => view('saas-base::filament.panels-topbar')
        );

        Filament::serving(function ()
        {
            // restrict RalphJSmit\MediaLibrary to tenants/teams
            MediaLibraryItem::saving(function (MediaLibraryItem $item)
            {
                $item->team_id = filament()->getTenant()->getKey();
            });

            MediaLibraryFolder::saving(function (MediaLibraryFolder $folder)
            {
                $folder->team_id = filament()->getTenant()->getKey();
            });

            MediaLibraryItem::addGlobalScope('team', function (Builder $query)
            {
                return $query->where('team_id', filament()?->getTenant()?->getKey()) ?? 0;
            });

            MediaLibraryFolder::addGlobalScope('team', function (Builder $query)
            {
                return $query->where('team_id', filament()->getTenant()->getKey());
            });

            if (auth()->check())
            {
                $timezone = validateTimezone(session('user_timezone')); // session is set in UserLoggedIn event

                DateTimePicker::configureUsing(fn (DateTimePicker $component) => $component->timezone($timezone));
                TextColumn::configureUsing(fn (TextColumn $column) => $column->timezone($timezone));
            }
        });
    }

    public static function get(): static
    {
        return filament(app(static::class)->getId());
    }

    public function getIsEnabled(): bool
    {
        return $this->isEnabled;
    }

    public function isEnabled(bool $value = true): static
    {
        $this->isEnabled = $value;

        return $this;
    }

    public function getZendeskWidgetEnabled(): bool
    {
        return $this->isZendeskWidgetEnabled;
    }

    public function enableZendeskWidget(bool $value = true): static
    {
        $this->isZendeskWidgetEnabled = $value;

        return $this;
    }

    public function getHelpEnabled(): bool
    {
        return $this->isHelpEnabled;
    }

    public function enableHelp(bool $value = true): static
    {
        $this->isHelpEnabled = $value;

        return $this;
    }

    public function getChangelogEnabled(): bool
    {
        return $this->isChangelogEnabled;
    }

    public function enableChangelog(bool $value = true): static
    {
        $this->isChangelogEnabled = $value;

        return $this;
    }

    public function canRegister(): bool
    {
        return $this->canRegister;
    }

    public function withRegistration(bool $value = true): static
    {
        $this->canRegister = $value;

        return $this;
    }

    public static function getCountryOptions()
    {
        $countryCodes = [
            'AL' => 'Albanien',
            'AT' => 'Oesterreich',
            'BE' => 'Belgien',
            'BG' => 'Bulgarien',
            'CH' => 'Schweiz',
            'CY' => 'Zypern',
            'CZ' => 'Tschechien',
            'DE' => 'Deutschland',
            'DK' => 'Dänemark',
            'EE' => 'Estland',
            'ES' => 'Spanien',
            'FI' => 'Finnland',
            'FR' => 'Frankreich',
            'GB' => 'United Kingdom',
            'GR' => 'Griechenland',
            'HR' => 'Kroatien',
            'HU' => 'Ungarn',
            'IE' => 'Irland',
            'IS' => 'Island',
            'IT' => 'Italien',
            'LI' => 'Liechtenstein',
            'LT' => 'Litauen',
            'LU' => 'Luxemburg',
            'LV' => 'Lettland',
            'MC' => 'Monaco',
            'MT' => 'Malta',
            'NL' => 'Niederlande',
            'PL' => 'Polen',
            'PT' => 'Portugal',
            'RO' => 'Rumänien',
            'SE' => 'Schweden',
            'SI' => 'Slowenien',
            'SK' => 'Slowakei',
            'NO' => 'Norwegen',
            'ME' => 'Montenegro',
            'MK' => 'Nordmazedonien',
            'RS' => 'Serbien',
            'TR' => 'Türkei',
            'UA' => 'Ukraine',
            'SM' => 'San Marino',
            'US' => 'USA'
        ];

        asort($countryCodes);

        return $countryCodes;
    }
}
