<?php

namespace Netzperfekt\SaasBase;

use Carbon\Carbon;
use Filament\Forms\Components\Component;
use Filament\Support\Assets\Css;
use Filament\Support\Assets\Js;
use Filament\Support\Facades\FilamentAsset;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasBase\Middleware\CheckBanned;
use Netzperfekt\SaasBase\Models\User;
use Spatie\LaravelPackageTools\Commands\InstallCommand;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;
use Spatie\Translatable\Facades\Translatable;

class SaasBaseServiceProvider extends PackageServiceProvider
{
    public function packageRegistered()
    {
        $this->app->register(EventServiceProvider::class);
    }

    public function bootingPackage()
    {
        $kernel = app(\Illuminate\Contracts\Http\Kernel::class);
        $kernel->appendMiddlewareToGroup('web', CheckBanned::class);

        activity()->disableLogging();
        if(config('saas-base.log_activities'))
        {
            activity()->enableLogging();
        }
    }

    public function packageBooted()
    {
        $this->addAssets();
        $this->addMacros();

        Translatable::fallback(
            fallbackAny: true,
        );

        Gate::define('viewPulse', function (User $user)
        {
            return $user->isSuperAdmin();
        });
    }

    public function configurePackage(Package $package): void
    {
        $package
            ->name(SaasBase::PACKAGE_NAME)
            ->hasRoute('web')
            ->hasViews()
            ->hasTranslations()
            ->hasConfigFile()

            ->hasMigrations([
                'create_user_types_table',
                'create_users_table',
                'create_teams_table',
                'create_team_user_table',
                'create_password_reset_tokens_table',
                // 'create_personal_access_tokens_table', // already in laravel/sanctum
                'create_activity_log_table',
                'create_failed_jobs_table',
                'create_settings_table',
                'create_media_table',
                'create_filament_media_library_table',
                'create_filament_media_library_folders_table',
            ])

            ->hasInstallCommand(function(InstallCommand $command)
            {
                $command
                    ->publishAssets()
                    ->publishMigrations()
                    ->publishConfigFile()
                    ->askToRunMigrations();
            });
    }

    private function addAssets()
    {
        FilamentAsset::register([
            Js::make('saas-base-js', __DIR__ . '/../resources/dist/saas-base.js'),
            Css::make('saas-base-css', __DIR__ . '/../resources/dist/saas-base.css')
        ], 'netzperfekt/saas-base');
    }

    private function addMacros()
    {
        /**
         * Sort models to their original list.
         * https://github.com/laravel/framework/issues/2327
         *
         * example:
         *       $ids = [50, 3] or collection
         *       $professionals = User::findMany($ids)
         *                                      ->sortToOriginal($ids);
         *
         * @param  \Illuminate\Contracts\Support\Arrayable|array  $ids
         * @return \Illuminate\Database\Eloquent\Collection
         */
        Collection::macro('sortToOriginal', function ($ids): Collection
        {
            $ids = $ids instanceof Arrayable ? $ids->toArray() : $ids;

            $models = array_flip($ids);

            foreach ($this as $model) {
                $models[ $model->id ] = $model;
            }

            return Collection::make(array_values($models));
        });

        Component::macro('htmlLabel', function($label, string $class = '')
        {
            if(! empty($class))
            {
                $label = '<span class="' . $class . '">' . $label . '</span>';
            }

            return $this->label(fn() => new HtmlString($label));
        });

        Component::macro('defaultRichEditor', function()
        {
            return
                $this->disableToolbarButtons([
                    'attachFiles',
                    'codeBlock',
                ])
                ->extraInputAttributes(['style' => 'min-height: 5rem; max-height: 25vh; overflow-y: auto;']);
        });

        Carbon::macro('formatWithTimezone', function (string $format, ?string $timeZoneOverride = null)
        {
            $timeZoneToFormat = 'UTC';
            if($timeZoneOverride)
            {
                $timeZoneToFormat = $timeZoneOverride;
            }
            else {
                $timeZoneToFormat = session('user_timezone', 'UTC');
            }

            return $this->setTimezone($timeZoneToFormat)->format($format);
        });
    }
}
