<?php

namespace Netzperfekt\SaasCms\Controllers;

use Illuminate\Http\Request;
use Illuminate\View\View;
use Netzperfekt\SaasCms\Data\CmsPageData;
use Netzperfekt\SaasCms\Enums\FieldType;
use Netzperfekt\SaasCms\Filament\Resources\CmsPageResource\CmsPageResource;
use Netzperfekt\SaasCms\Filament\Resources\CmsTemplateResource\CmsTemplateResource;
use Netzperfekt\SaasCms\Models\CmsPage;
use Netzperfekt\SaasCms\Models\CmsTemplate;
use Netzperfekt\SaasCms\SaasCms;
use RalphJSmit\Filament\MediaLibrary\Media\Models\MediaLibraryItem;

class CmsPageController extends Controller
{
    public function index(Request $request): View
    {
        $cmsPage = CmsPage::active()->homepage()->first();

        return $this->getPage($request, $cmsPage);
    }

    public function showPage(Request $request, string $slug): View
    {
        $cmsPage = CmsPage::active()->withSlug($slug)->first();

        return $this->getPage($request, $cmsPage);
    }

    public function getPage(Request $request, ?CmsPage $cmsPage): View
    {
        if(empty($cmsPage))
        {
            abort(404);
        }

        $fields = (object)$this->resolveFields($cmsPage->contents, $cmsPage->template);

        $pageData = CmsPageData::from(
            [
                'page'           => $cmsPage,
                'template'       => $cmsPage->template,
                'fields'         => $fields,
                'slug'           => $cmsPage->slug,
                'title'          => $cmsPage->title,
                'seoTitle'       => $cmsPage->seo_title,
                'seoDescription' => $cmsPage->seo_description,
            ]
        );

        return view(SaasCms::PACKAGE_NAME . '::cmspage', ['data' => $pageData]);
    }

    private function resolveFields(?array $fields, CmsTemplate $template): array
    {
        $data = [];
        if(empty($fields))
        {
            return [];
        }

        foreach($fields as $fieldKey => $fieldValue)
        {
            $fieldType = null;
            $templateFields = [];

            foreach($template->fields as $templateField)
            {
                if($templateField['type'] == 'group')
                {
                    foreach($templateField['data']['fields'] as $groupField)
                    {
                        $templateFields[] = $groupField;
                    }
                }
                else
                {
                    $templateFields[] = $templateField;
                }
            }

            $fieldTypesFiltered = array_filter($templateFields, function($val) use ($fieldKey)
            {
                return $val['data']['id'] == $fieldKey;
            });
            if(count($fieldTypesFiltered) > 0)
            {
                $fieldType = FieldType::from(array_shift($fieldTypesFiltered)['data']['type']);
            }

            if($fieldType)
            {
                $data[$fieldKey] = $fieldType->resolveValue($fieldValue);
            }
        }

        return $data;
    }

    public function previewTemplate(Request $request, string $id)
    {
        $template = CmsTemplate::find($id);
        if($template->template_type !== CmsTemplateResource::TEMPLATE_TYPE_PAGE)
        {
            abort(403, 'template preview is only possible for template types "page"');
        }

        $fields = [];
        foreach($template->fields ?? [] as $field)
        {
            if($field['type'] == 'group')
            {
                foreach($field['data']['fields'] as $groupField)
                {
                    $fieldEnum = FieldType::from($groupField['data']['type']);
                    $fields[$groupField['data']['id']] = $fieldEnum->getDummyContent();
                }
            }
            else
            {
                $fieldEnum = FieldType::from($field['data']['type']);
                $fields[$field['data']['id']] = $fieldEnum->getDummyContent();
            }
        }

        $fields = (object)$fields;

        $pageData = CmsPageData::from(
            [
                'page'           => null,
                'template'       => $template,
                'fields'         => $fields,
                'slug'           => '',
                'title'          => '- template preview -',
                'seoTitle'       => '',
                'seoDescription' => '',
            ]
        );

        return view(SaasCms::PACKAGE_NAME . '::cmstemplate', ['data' => $pageData]);
    }
}
