<?php

namespace Netzperfekt\SaasCms\Filament\Resources\CmsPageResource;

use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\ActionSize;
use Filament\Support\Enums\FontWeight;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;
use Netzperfekt\SaasCms\Filament\Exporter\CmsPageExporter;
use Netzperfekt\SaasCms\Filament\Exporter\CmsPageImporter;
use Netzperfekt\SaasCms\Filament\Exporter\ExportAction;
use Netzperfekt\SaasCms\Filament\Exporter\ImportAction;
use Netzperfekt\SaasCms\Filament\Resources\CmsPageResource\Pages\CreateCmsPage;
use Netzperfekt\SaasCms\Filament\Resources\CmsPageResource\Pages\EditCmsPage;
use Netzperfekt\SaasCms\Filament\Resources\CmsPageResource\Pages\ListCmsPages;
use Netzperfekt\SaasCms\Filament\Resources\CmsTemplateResource\CmsTemplateResource;
use Netzperfekt\SaasCms\Models\CmsPage;
use Netzperfekt\SaasCms\Models\CmsTemplate;
use Netzperfekt\SaasCms\SaasCms;
use RalphJSmit\Filament\MediaLibrary\Tables\Columns\MediaColumn;

class CmsPageResource extends Resource
{
    protected static ?string $model = CmsPage::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = 50;

    public static function getNavigationGroup(): ?string
    {
        return __('cms-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return __('cms-page');
    }

    public static function getPluralModelLabel(): string
    {
        return __('cms-pages');
    }

    public static function form(Form $form): Form
    {
        $cmsGroupDefinition = getCmsGroups();
        $thisPackage = SaasCms::get();

        return $form
            ->schema([
                Section::make('')
                    ->label(__('user-data'))
                    ->columnSpan(2)
                    ->columns(3)
                    ->schema([
                        TextInput::make('title')
                            ->label(__('cms-title'))
                            ->required()
                            ->maxLength(255)
                            ->columnSpan(2)
                            ->live()
                            ->afterStateUpdated(function ($get, $set, ?string $state)
                            {
                                if($get('is_homepage'))
                                {
                                    $set('slug', '/');
                                }
                                elseif(empty($get('slug'))) {
                                    $set('slug', Str::slug($state));
                                }

                                if(empty($get('seo_title'))) {
                                    $set('seo_title', $state);
                                }
                            }),

                        TextInput::make('slug')
                            ->label(__('cms-slug'))
                            ->required(fn($get) => ! $get('is_homepage'))
                            ->visible(fn($get) => ! $get('is_homepage'))
                            ->maxLength(255)
                            ->columnSpan(1)
                            ->unique(ignoreRecord: true),

                        Grid::make(__('cms-contents'))
                            ->columnSpanFull()
                            ->schema(fn($get) => CmsTemplate::find($get('template_id'))?->getForm() ?? [])
                            ->default([]),
                    ]),

                Section::make(__('options'))
                    ->columnSpan(1)
                    ->schema([
                        Toggle::make('is_active')
                            ->label(__('cms-active'))
                            ->columnSpan(1)
                            ->default(true)
                            ->offColor('danger')
                            ->onColor('success'),

                        Toggle::make('is_homepage')
                            ->label(__('cms-homepage'))
                            ->columnSpan(1)
                            ->live()
                            ->default(false),

                        Select::make('template_id')
                            ->label(__('cms-template'))
                            ->columnSpan(1)
                            ->relationship(
                                'template',
                                'title',
                                modifyQueryUsing: fn (Builder $query) => $query->where('template_type', CmsTemplateResource::TEMPLATE_TYPE_PAGE)
                            )
                            ->required()
                            ->live(),

                        Select::make('group')
                            ->label(__('cms-group'))
                            ->columnSpan(1)
                            ->required()
                            ->selectablePlaceholder(fn() => ! $thisPackage->getForTenants())
                            ->default(function() use ($cmsGroupDefinition, $thisPackage)
                            {
                                if($thisPackage->getForTenants())
                                {
                                    return $cmsGroupDefinition->tenantGroups[$thisPackage->getForTenantsPackage()];
                                }
                                else
                                {
                                    return 'default';
                                }

                                return null;
                            })
                            ->options(function() use ($cmsGroupDefinition, $thisPackage)
                            {
                                if($thisPackage->getForTenants())
                                {
                                    $tenantGroup = $cmsGroupDefinition->tenantGroups[$thisPackage->getForTenantsPackage()];
                                    return [
                                        $tenantGroup => __('cms-group-' . $tenantGroup)
                                    ];
                                }
                                else
                                {
                                    return $cmsGroupDefinition->groups;
                                }
                            })
                            ->columns(1)
                    ]),

                TextInput::make('seo_title')
                    ->label(__('cms-seotitle'))
                    ->columnSpanFull()
                    ->required()
                    ->maxLength(70),

                Textarea::make('seo_description')
                    ->label(__('cms-seodescription'))
                    ->rows(3)
                    ->minLength(120)
                    ->maxLength(300)
                    ->columnSpan(3)
                    ->live(debounce: 250)
                    ->hint(fn ($state, $component) => strlen($state ?? '') . ' / ' . $component->getMaxLength()),

                /* TODO als FieldType definieren
                MediaPicker::make('image_ids')
                    ->label(__('cms-select-images'))
                    ->multiple()
                    ->reorderable()
                    ->columnSpan(2)
                */
            ]);
    }

    public static function table(Table $table): Table
    {
        $cmsGroupDefinition = getCmsGroups();
        $thisPackage = SaasCms::get();

        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('cms-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                MediaColumn::make('thumbnail')
                    ->label(__('cms-thumbnail')),

                Tables\Columns\TextColumn::make('slug')
                    ->label(__('cms-slug'))
                    ->searchable(),

                Tables\Columns\TextColumn::make('template.title')
                    ->label(__('cms-template'))
                    ->badge()
                    ->color(Color::Orange),

                Tables\Columns\TextColumn::make('group')
                    ->label(__('cms-group'))
                    ->badge()
                    ->formatStateUsing(function (string $state) use ($cmsGroupDefinition, $thisPackage)
                    {
                        if($thisPackage->getForTenants())
                        {
                            return __('cms-group-' . $cmsGroupDefinition->tenantGroups[$thisPackage->getForTenantsPackage()])
                                    ?? '(' . $state . ')';
                        }
                        else
                        {
                            return $cmsGroupDefinition->groups[$state] ?? '(' . $state . ')';
                        }
                    }),

                Tables\Columns\ToggleColumn::make('is_homepage')
                    ->disabled()
                    ->label(__('cms-homepage')),

                Tables\Columns\ToggleColumn::make('is_active')
                    ->disabled()
                    ->label(__('cms-active'))
                    ->offColor('danger')
                    ->onColor('success'),
            ])

            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('group')
                    ->label(__('cms-group'))
                    ->options($cmsGroupDefinition->groups)
            ])

            ->headerActions([
                ExportAction::make()
                    ->label('Export')
                    ->exporter(CmsPageExporter::class),

                ImportAction::make()
                    ->label('Import')
                    ->importer(CmsPageImporter::class)
            ])

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),

                Tables\Actions\Action::make('preview')
                    ->iconButton()
                    ->tooltip(__('cms-preview'))
                    ->icon('fas-bolt')
                    ->size(ActionSize::Medium)
                    ->color('info')
                    ->url(fn($record) => $record->is_homepage ? route('cms.home') : route('cms.page', $record->slug))
                    ->openUrlInNewTab()
                    ->visible(fn($record) => $record->slug),

                Tables\Actions\ReplicateAction::make()
                    ->iconButton()
                    ->tooltip('Duplizieren')
                    ->beforeReplicaSaved(function ($replica)
                    {
                        $replica->title .= ' - ' . __('cms-copy');
                        $replica->slug = Str::slug($replica->title);
                    }),

                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\ExportBulkAction::make()->exporter(CmsPageExporter::class),

                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListCmsPages::route('/'),
            'create' => CreateCmsPage::route('/create'),
            'edit' => EditCmsPage::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ])
            ->orderBy('title');
    }
}
