<?php

namespace Netzperfekt\SaasCms\Filament\Resources\CmsTemplateResource;

use Filament\Forms\Components\Builder;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\ActionSize;
use Filament\Support\Enums\FontWeight;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;
use Netzperfekt\SaasCms\Enums\FieldType;
use Netzperfekt\SaasCms\Filament\Exporter\CmsTemplateExporter;
use Netzperfekt\SaasCms\Filament\Exporter\CmsTemplateImporter;
use Netzperfekt\SaasCms\Filament\Exporter\ExportAction;
use Netzperfekt\SaasCms\Filament\Exporter\ImportAction;
use Netzperfekt\SaasCms\Filament\Resources\CmsTemplateResource\Pages\CreateCmsTemplate;
use Netzperfekt\SaasCms\Filament\Resources\CmsTemplateResource\Pages\EditCmsTemplate;
use Netzperfekt\SaasCms\Filament\Resources\CmsTemplateResource\Pages\ListCmsTemplates;
use Netzperfekt\SaasCms\Models\CmsTemplate;
use Riodwanto\FilamentAceEditor\AceEditor;

class CmsTemplateResource extends Resource
{
    protected static ?string $model = CmsTemplate::class;
    protected static bool $isScopedToTenant = false;

    protected static ?string $navigationIcon = 'heroicon-o-squares-2x2';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = 50;

    public final const string TEMPLATE_TYPE_PAGE = 'page';
    public final const string TEMPLATE_TYPE_PARTIAL = 'partial';

    public static function getNavigationGroup(): ?string
    {
        return __('cms-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::count();
    }

    public static function getModelLabel(): string
    {
        return __('cms-template');
    }

    public static function getPluralModelLabel(): string
    {
        return __('cms-templates');
    }

    public static function form(Form $form): Form
    {
        $cmsTypeDefinitions = getCmsTypes();
        $templateTypeOptions = self::getDefaultTemplateTypes() + $cmsTypeDefinitions->types;

        return $form
            ->columns(5)
            ->schema([
                TextInput::make('title')
                    ->label(__('cms-title'))
                    ->required()
                    ->maxLength(255)
                    ->columnSpan(1)
                    ->autofocus()
                    ->live()
                    ->afterStateUpdated(function ($get, $set, ?string $state)
                    {
                        if(empty($get('slug'))) {
                            $set('key', Str::slug($state));
                        }
                    }),

                TextInput::make('description')
                    ->label(__('cms-description'))
                    ->maxLength(255)
                    ->columnSpan(2)
                    ->autofocus(),

                TextInput::make('key')
                    ->label(__('cms-key'))
                    ->required()
                    ->maxLength(255)
                    ->columnSpan(1)
                    ->unique(ignoreRecord: true),

                Select::make('template_type')
                    ->label(__('cms-template-type'))
                    ->columnSpan(1)
                    ->required()
                    ->options($templateTypeOptions)
                    ->default(self::TEMPLATE_TYPE_PAGE),

                Tabs::make('Tabs')
                    ->columnSpanFull()
                    ->tabs([
                        Tabs\Tab::make(__('cms-template'))
                            ->schema([

                                // https://filamentphp.com/plugins/riodewanto-ace-editor
                                // NE 04.03.25 - not compatible with laravel 12, temp. removed
                                /*
                                AceEditor::make('template')
                                    ->mode('html')
                                    ->placeholder('')
                                    ->height('550px')
                                    ->editorOptions([
                                        'fontSize' => '13px',
                                    ])
                                    ->rows(4)
                                    ->grow(true)
                                    ->label(__('cms-contents'))
                                    ->required()
                                */
                                Textarea::make('template')
                                    ->placeholder('')
                                    ->rows(10)
                                    ->autosize(true)
                                    ->label(__('cms-contents'))
                                    ->required()

                            ]),

                        Tabs\Tab::make(__('cms-fields'))
                            ->schema([
                                Builder::make('fields')
                                    ->label(__('cms-fields'))
                                    ->blockNumbers(false)
                                    ->columns(5)
                                    ->collapsible()
                                    ->cloneable()
                                    ->blocks(self::getCmsFieldBuilderBlocks()),

                            ])
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        $cmsTypeDefinitions = getCmsTypes();
        $templateTypeOptions = self::getDefaultTemplateTypes() + $cmsTypeDefinitions->types;

        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('cms-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('description')
                    ->label(__('cms-description'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('key')
                    ->label(__('cms-key'))
                    ->badge()
                    ->color(Color::Yellow)
                    ->searchable(),

                Tables\Columns\TextColumn::make('template_type')
                    ->label(__('cms-template-type'))
                    ->formatStateUsing(fn($state) => $templateTypeOptions[$state] ?? $state)
                    ->badge()
            ])
            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),

                Tables\Actions\Action::make('preview')
                    ->iconButton()
                    ->tooltip(__('cms-preview'))
                    ->icon('fas-bolt')
                    ->size(ActionSize::Medium)
                    ->color('info')
                    ->url(fn($record) => route('cms.template', $record->id))
                    ->openUrlInNewTab()
                    ->visible(fn($record) => $record->template_type == self::TEMPLATE_TYPE_PAGE),

                Tables\Actions\ReplicateAction::make()
                    ->iconButton()
                    ->tooltip('Duplizieren')
                    ->beforeReplicaSaved(function ($replica)
                    {
                        $replica->title .= ' - ' . __('cms-copy');
                    }),

                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\ExportBulkAction::make()->exporter(CmsTemplateExporter::class),

                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            ->headerActions([
                ExportAction::make()
                    ->label('Export')
                    ->exporter(CmsTemplateExporter::class),

                ImportAction::make()
                    ->label('Import')
                    ->importer(CmsTemplateImporter::class)
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('template_type')
                    ->options($templateTypeOptions)
                    ->label(__('cms-template-type'))
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);

    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListCmsTemplates::route('/'),
            'create' => CreateCmsTemplate::route('/create'),
            'edit' => EditCmsTemplate::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ])
            ->orderBy('title');
    }

    public static function getCmsFieldBuilderBlocks()
    {
        return array_merge(
            self::getCmsFieldBuilderFields(),

            [
                Builder\Block::make('group')
                    ->label(fn ($state) => ! $state ? __('cms-group') : ($state['title'] ?? '-'))
                    ->columns(5)
                    ->schema([
                        TextInput::make('title')
                            ->label('')
                            ->placeholder(__('cms-group-title'))
                            ->columnSpan(1)
                            ->live()
                            ->extraAttributes(['class' => 'font-semibold']),

                        TextInput::make('description')
                            ->label('')
                            ->placeholder(__('cms-group-description'))
                            ->columnSpan(2),

                        Builder::make('fields')
                            ->label('')
                            ->columnSpanFull()
                            ->blockNumbers(false)
                            ->collapsible()
                            ->cloneable()
                            ->blocks(self::getCmsFieldBuilderFields()),
                    ])
            ]
        );
    }

    public static function getCmsFieldBuilderFields()
    {
        return [
            Builder\Block::make('field')
                ->label(fn ($state) => ! $state ? __('cms-field') : ($state['title'] ?? '-'))
                ->extraAttributes(['class' => 'font-semibold'])
                ->columns(4)
                ->schema([
                    TextInput::make('id')
                        ->label(__('cms-field-id'))
                        ->required(),

                    TextInput::make('title')
                        ->label(__('cms-field-title'))
                        ->live()
                        ->required(),

                    Select::make('type')
                        ->label(__('cms-field-type'))
                        ->options(FieldType::class)
                        ->required(),

                    Toggle::make('required')
                        ->label(__('cms-field-required')),
                ])
        ];
    }

    private static function getDefaultTemplateTypes(): array
    {
        return [
            self::TEMPLATE_TYPE_PAGE    => __('cms-page'),
            self::TEMPLATE_TYPE_PARTIAL => __('cms-partial'),
        ];
    }
}
