<?php

namespace Netzperfekt\SaasCms\Filament\Resources\CmsTreeResource;

use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\ToggleButtons;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Enums\FontWeight;
use Filament\Tables;
use Filament\Tables\Table;
use Netzperfekt\SaasCms\Filament\Exporter\CmsTreeExporter;
use Netzperfekt\SaasCms\Filament\Exporter\CmsTreeImporter;
use Netzperfekt\SaasCms\Filament\Exporter\ExportAction;
use Netzperfekt\SaasCms\Filament\Exporter\ImportAction;
use Netzperfekt\SaasCms\Filament\Resources\CmsTreeResource\Pages\CreateCmsTree;
use Netzperfekt\SaasCms\Filament\Resources\CmsTreeResource\Pages\EditCmsTree;
use Netzperfekt\SaasCms\Filament\Resources\CmsTreeResource\Pages\ListCmsTree;
use Netzperfekt\SaasCms\Models\CmsPage;
use Netzperfekt\SaasCms\Models\CmsTree;
use Saade\FilamentAdjacencyList\Forms\Components\AdjacencyList;

class
CmsTreeResource extends Resource
{
    protected static ?string $model = CmsTree::class;

    protected static ?string $navigationIcon = 'heroicon-o-list-bullet';
    protected static ?int $navigationSort = 51;

    public static function getNavigationGroup(): ?string
    {
        return __('cms-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::count();
    }

    public static function getModelLabel(): string
    {
        return __('cms-tree');
    }

    public static function getPluralModelLabel(): string
    {
        return __('cms-trees');
    }

    public static function form(Form $form): Form
    {
        $cmsGroupDefinition = getCmsGroups();

        $cmsTypeDefinition = getCmsTypes();
        $additionalTypes = $cmsTypeDefinition->types;
        $additionalTypeModels = $cmsTypeDefinition->models;

        return $form
            ->columns(4)
            ->schema([
                TextInput::make('title')
                    ->label(__('cms-title'))
                    ->required()
                    ->columns(2)
                    ->maxLength(255),

                Select::make('group')
                    ->label(__('cms-group'))
                    ->required()
                    ->options($cmsGroupDefinition->groups)
                    ->columns(1)
                    ->live(),

                Select::make('position')
                    ->label(__('cms-position'))
                    ->required()
                    ->options(fn($get) => $cmsGroupDefinition->positions[$get('group')] ?? [])
                    ->columns(1)
                    ->live(),

                // https://filamentphp.com/plugins/saade-adjacency-list
                AdjacencyList::make('items')
                    ->label(__('cms-tree-items'))
                    ->columnSpanFull()
                    ->labelKey('label')
                    ->childrenKey('children')
                    ->maxDepth(2)
                    ->orderColumn('sort')
                    ->addAction(fn (Action $action): Action =>
                                    $action
                                        ->label(__('add-entry'))
                                        ->modalHeading(__('add-entry')))

                    ->form(function($get) use ($additionalTypes, $additionalTypeModels)
                    {
                        return [
                            Grid::make(3)->schema([
                                TextInput::make('label')
                                    ->columns(1)
                                    ->autofocus()
                                    ->required(),

                                ToggleButtons::make('type')
                                    ->label(__('type'))
                                    ->default('cms')
                                    ->inline()
                                    ->live()
                                    ->options([
                                        'cms' => __('type-cms'),
                                        'url' => __('type-url'),
                                        'navigation' => __('type-navigation'),
                                        'action' => __('type-action')
                                    ] + $additionalTypes)
                                    ->icons([
                                        'cms' => 'heroicon-o-rectangle-stack',
                                        'url' => 'heroicon-o-link',
                                        'navigation' => 'heroicon-o-folder-arrow-down',
                                        'action' => 'heroicon-o-play',
                                    ]),

                                Select::make('cmspage')
                                    ->label(__('cms-page'))
                                    ->visible(fn($get) => $get('type') == 'cms')
                                    ->required()
                                    ->columns(1)
                                    ->options(CmsPage::all()->where('group', $get('group'))->sortBy('title')->pluck('title', 'id'))
                                    ->searchable(),

                                TextInput::make('url')
                                    ->label(__('url'))
                                    ->visible(fn($get) => $get('type') == 'url')
                                    ->required(),

                                Select::make('action')
                                    ->label(__('cms-action'))
                                    ->visible(fn($get) => $get('type') == 'action')
                                    ->required()
                                    ->columns(1)
                                    ->options([
                                        'register' => __('cms-action-register'),
                                        'login' => __('cms-action-login'),
                                    ]),

                                Select::make('cmspage')
                                    ->label(function($get) use ($additionalTypes)
                                    {
                                        return $additionalTypes[$get('type')];
                                    })
                                    ->visible(fn($get) => $get('type') != 'cms' && $get('type') != 'url' && $get('type') != 'navigation' && $get('type') != 'action')
                                    ->required()
                                    ->columns(1)
                                    ->options(function($get) use ($additionalTypeModels)
                                    {
                                        $additionalType = $additionalTypeModels[$get('type')];

                                        return $additionalType['model']::all()
                                            ->sortBy($additionalType['name'])
                                            ->pluck($additionalType['name'], $additionalType['id']);
                                    })
                                    ->searchable(),
                            ])
                        ];
                    })
            ]);
    }

    public static function table(Table $table): Table
    {
        $cmsGroupDefinition = getCmsGroups();

        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('cms-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('group')
                    ->label(__('cms-group'))
                    ->badge()
                    ->color('warning')
                    ->formatStateUsing(fn (string $state): string => $cmsGroupDefinition->groups[$state] ?? '(' . $state . ')'),

                Tables\Columns\TextColumn::make('position')
                    ->label(__('cms-position'))
                    ->badge()
                    ->color('info')
                    ->formatStateUsing(fn ($record, $state) => $cmsGroupDefinition->positions[$record->group][$state] ?? '(' . $state . ')'),

            ])
            ->filters([
                Tables\Filters\SelectFilter::make('group')
                    ->label(__('cms-group'))
                    ->options($cmsGroupDefinition->groups)
            ])

            ->headerActions([
                ExportAction::make()
                    ->label('Export')
                    ->exporter(CmsTreeExporter::class),

                ImportAction::make()
                    ->label('Import')
                    ->importer(CmsTreeImporter::class)
            ])

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),

                Tables\Actions\ReplicateAction::make()
                    ->iconButton()
                    ->tooltip('Duplizieren')
                    ->beforeReplicaSaved(function ($replica)
                    {
                        $replica->title .= ' - ' . __('cms-copy');
                    }),

                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\ExportBulkAction::make()->exporter(CmsTreeExporter::class),

                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListCmsTree::route('/'),
            'create' => CreateCmsTree::route('/create'),
            'edit' => EditCmsTree::route('/{record}/edit'),
        ];
    }
}
