<?php

namespace Netzperfekt\SaasCms\Models;

use Filament\Forms\Components\Component;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle;
use Filament\Support\Enums\MaxWidth;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Netzperfekt\SaasBase\Traits\HasDefaultScopes;
use Netzperfekt\SaasBase\Traits\HasGlobalModelObserver;
use Netzperfekt\SaasCms\Enums\FieldType;
use Netzperfekt\SaasCms\Events\GetFaqGroupsEvent;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;

class CmsTemplate extends Model implements HasMedia
{
    use InteractsWithMedia, SoftDeletes;
    use HasGlobalModelObserver;
    use HasDefaultScopes;

    protected $fillable = [
        'team_id',
        'template_type',
        'title',
        'key',
        'description',
        'template',
        'fields',
    ];

    protected $casts = [
        'fields' => 'array'
    ];

    public function page(): BelongsTo
    {
        return $this->belongsTo(CmsPage::class);
    }

    // TODO refactor woanders hin - gehört hier nicht so richtig rein...
    public function getForm()
    {
        $blocks = [];
        $groups = [];

        foreach($this->fields ?? [] as $field)
        {
            $blockType = $field['type'];

            if($blockType == 'group')
            {
                $groupTitle = $field['data']['title'];
                $groupDescription = $field['data']['description'] ?? '';

                foreach($field['data']['fields'] as $groupField)
                {
                    if( ! array_key_exists($groupTitle, $groups))
                    {
                        $groups[$groupTitle] = Section::make($groupTitle)
                            ->description($groupDescription)
                            ->compact()
                            ->collapsible();
                    }

                    $groups[$groupTitle]->schema(
                        array_merge($groups[$groupTitle]->getChildComponents(), [ $this->addField($groupField) ])
                    );
                }
            }
            else {
                $blocks[] = $this->addField($field);
            }
        }

        return $blocks + $groups;
    }

    private function addField($field)
    {
        $fieldId = $field['data']['id'];
        $fieldType = $field['data']['type'];
        $fieldTitle = $field['data']['title'];
        $fieldRequired = $field['data']['required'];

        $componentClass = FieldType::from($fieldType)->getFormComponent();

        if (class_exists($componentClass)) {
            // TODO required() klappt hier noch nicht so richtig - achtung: nicht für GRID!

            /* @var Component */
            $component = $componentClass::make('contents.' . $fieldId)
                ->label($fieldTitle);

            if (FieldType::File->equals($fieldType))
            {
                $component->maxWidth(MaxWidth::ExtraSmall);
            }
            elseif (FieldType::Link->equals($fieldType))
            {
                $component->compact()
                    ->columns(4)
                    ->statePath('contents.' . $fieldId)
                    ->heading($fieldTitle);

                $component->schema([
                    TextInput::make('title')
                        ->label(__('cms-field-type-link-title'))
                        ->required(),

                    TextInput::make('subtitle')
                        ->label(__('cms-field-type-link-subtitle')),

                    TextInput::make('url')
                        ->label(__('cms-field-type-link-url'))
                        ->required(),

                    Toggle::make('target')
                        ->label(__('cms-field-type-link-openblank'))

                ]);
            }
            elseif (FieldType::InfoBlock->equals($fieldType))
            {
                $component->compact()
                    ->columns(3)
                    ->statePath('contents.' . $fieldId)
                    ->heading($fieldTitle);

                $component->schema([
                    TextInput::make('icon')
                        ->label(__('cms-field-type-infoblock-icon'))
                        ->maxWidth(40),

                    TextInput::make('title')
                        ->label(__('cms-field-type-infoblock-title'))
                        ->maxWidth(40)
                        ->required(),

                    Textarea::make('text')
                        ->maxWidth(500)
                        ->label(__('cms-field-type-infoblock-text'))

                ]);
            }
            elseif (FieldType::Faq->equals($fieldType))
            {
                $faqTemplates = CmsTemplate::orderBy('title')->get();
                $faqGroups = array_filter(event(new GetFaqGroupsEvent()));
                $faqGroups = array_shift($faqGroups);

                $component->compact()
                    ->columns(3)
                    ->statePath('contents.' . $fieldId)
                    ->heading($fieldTitle);

                $component->schema([
                    Select::make('group')
                        ->label(__('cms-field-type-faq-faqgroup'))
                        ->options($faqGroups['groups']->pluck('title', 'id'))
                        ->required(),

                    Select::make('template')
                        ->label(__('cms-field-type-faq-template'))
                        ->options($faqTemplates->pluck('title', 'id'))
                        ->required(),

                ]);
            }
        }

        return $component;
    }
}
