<?php

namespace Netzperfekt\SaasCms\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Route;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasBase\Traits\HasDefaultScopes;
use Netzperfekt\SaasBase\Traits\HasGlobalModelObserver;
use Netzperfekt\SaasCms\Enums\CmsLinkType;
use Netzperfekt\SaasCms\Events\GetCmsLinkTypesEvent;

class CmsTree extends Model
{
    use HasGlobalModelObserver;
    use HasDefaultScopes;

    protected $table = 'cms_tree';

    protected $casts = [
        'items' => 'array'
    ];

    protected $fillable = [
        'team_id',
        'title',
        'group',
        'position',
        'items'
    ];

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public static function getNavigation(string $position, string $group = 'default')
    {
        $cmsTree = CmsTree::where('group', $group)->where('position', $position)->first();

        return $cmsTree?->resolveNavigation() ?? [];
    }

    public function resolveNavigation()
    {
        return $this->resolveCmsTree($this?->items ?? []);
    }

    private function resolveCmsTree(array $cmsTree)
    {
        $results = [];

        foreach($cmsTree as $item)
        {
            $cmsPage = null;
            $url = '';
            $itemType = $item['type'] ?? 'cms';
            $itemAction = '';
            $itemEnabled = true;

            if($itemType == 'cms')
            {
                $cmsPage = CmsPage::find($item['cmspage']);
                if($cmsPage)
                {
                    $url = route('cms.page', [ $cmsPage->slug ]);
                }
            }
            elseif($itemType == 'url')
            {
                $url = $item['url'];
            }
            elseif($itemType == 'action')
            {
                $itemAction = $item['action'];

                if($itemAction == 'register')
                {
                    $itemEnabled = Route::has('filament.app.auth.register');
                    $url = $itemEnabled ? route('filament.app.auth.register') : '';
                }
                elseif($itemAction == 'login')
                {
                    $itemEnabled = Route::has('filament.app.auth.login');
                    $url = $itemEnabled ? route('filament.app.auth.login') : '';
                }
            }
            else
            {
                $callbackResults = array_filter(event(new GetCmsLinkTypesEvent(CmsLinkType::Route)));
                $callbackResults = array_shift($callbackResults);

                try {
                    $modelInstance = (new $callbackResults['model'])->find($item['cmspage']);
                }
                catch(\Exception)
                {
                    //
                }

                $cmsPage = $modelInstance;
                $url = route($callbackResults['route'], [ $cmsPage->slug ]);
            }

            $results[$item['label']] = [
                'type'     => $itemType,
                'enabled'  => $itemEnabled,
                'cmspage'  => $cmsPage,
                'url'      => $url,
                'action'   => $itemAction,
                'children' => $this->resolveCmsTree($item['children'])
            ];
        }

        return $results;
    }
}
