<?php

namespace Netzperfekt\SaasCodoc\Enums;

use Filament\Support\Contracts\HasIcon;
use Filament\Support\Contracts\HasLabel;
use Illuminate\Support\Facades\Auth;
use Netzperfekt\SaasBase\Models\User;

enum OrganisationType: int implements HasLabel, HasIcon
{
    case None            =  0;

    case Customer        =  1;
    case Employee        =  2;

    case Supplier        = 10;
    case ServiceProvider = 11;
    case LocalSoftware   = 12;

    case Stakeholder     = 20;
    case Tax             = 21;
    case Legal           = 22;
    case Banking         = 23;
    case Insurance       = 24;
    case Medical         = 25;
    case Landlord        = 26;

    case Consultant      = 30;

    // *** ACHTUNG: Subtypen auch in GetTypesForUser rausnehmen/beachten! ***
    case Partner         = 40;
    case Partner_Current = 401;
    case Partner_Former  = 402;

    case Family          = 41;
    case Family_Children = 410;
    case Family_Parents  = 411;
    case Family_Sibling  = 412;
    case Family_Other    = 413;

    case Friends         = 42;
    case PrivateOther    = 43;

    case Association     = 50;

    case OwnProduct      = 90;
    case Company         = 91;
    case OwnOrganisation = 99;

    public const UserTypeBusiness = 'default';
    public const UserTypePrivate  = 'private';

    public function getLabel(): ?string
    {
        return match ($this) {
            self::None            => '---',

            self::Customer        => __('codoc-enum-organisation-customer'),
            self::Supplier        => __('codoc-enum-organisation-supplier'),
            self::Stakeholder     => __('codoc-enum-organisation-stakeholder'),
            self::Tax             => __('codoc-enum-organisation-tax'),
            self::Legal           => __('codoc-enum-organisation-legal'),
            self::Banking         => __('codoc-enum-organisation-bank'),
            self::Insurance       => __('codoc-enum-organisation-insurance'),
            self::Medical         => __('codoc-enum-organisation-medical'),
            self::Consultant      => __('codoc-enum-organisation-consultant'),
            self::ServiceProvider => __('codoc-enum-organisation-serviceprovider'),
            self::LocalSoftware   => __('codoc-enum-organisation-localsoftware'),
            self::Employee        => __('codoc-enum-organisation-employee'),
            self::OwnOrganisation => self::isPrivateUser() ? __('codoc-enum-organisation-myself') : __('codoc-enum-organisation-ownorganisation'),
            self::OwnProduct      => __('codoc-enum-organisation-ownproduct'),
            self::Company         => __('codoc-enum-organisation-company'),
            self::Association     => __('codoc-enum-organisation-association'),

            self::Partner         => __('codoc-enum-organisation-partner'),
            self::Partner_Current => __('codoc-enum-organisation-partner-current'),
            self::Partner_Former  => __('codoc-enum-organisation-partner-former'),

            self::Family          => __('codoc-enum-organisation-family'),
            self::Family_Children => __('codoc-enum-organisation-family-children'),
            self::Family_Parents  => __('codoc-enum-organisation-family-parents'),
            self::Family_Sibling  => __('codoc-enum-organisation-family-sibling'),
            self::Family_Other    => __('codoc-enum-organisation-family-other'),

            self::Friends         => __('codoc-enum-organisation-friends'),
            self::PrivateOther    => __('codoc-enum-organisation-private-other'),
            self::Landlord        => __('codoc-enum-organisation-landlord'),

            default               => '* unknown type: ' . $this->value . ' *'
        };
    }

    public function getIcon(): ?string
    {
        return match ($this) {
            self::None            => '',

            self::Customer        => 'fas-people-arrows',
            self::Supplier        => 'fas-person-walking-arrow-right',
            self::Stakeholder     => 'fas-people-group',
            self::Tax             => 'fas-coins',
            self::Banking         => 'fas-credit-card',
            self::Insurance       => 'fas-house-crack',
            self::Medical         => 'fas-user-doctor',
            self::Legal           => 'fas-section',
            self::Consultant      => 'fas-person-walking-luggage',
            self::ServiceProvider => 'fas-cloud',
            self::LocalSoftware   => 'fas-database',
            self::Employee        => 'fas-user',
            self::OwnOrganisation => self::isPrivateUser() ? 'fas-crown' : 'fas-building',
            self::OwnProduct      => 'fas-box',
            self::Company         => 'fas-handshake',
            self::Association     => 'fas-users-rectangle',
            self::Partner         => 'fas-people-arrows',
            self::Family          => 'fas-people-group',
            self::Friends         => 'fas-people-group',
            self::PrivateOther    => 'fas-people-group',
            self::Landlord        => 'fas-person-shelter',

            default               => '* unknown type: ' . $this->value . ' *'
        };
    }

    public function providesServices(): ?string
    {
        return match ($this) {
            self::None            => false,

            self::Customer        => false,
            self::Supplier        => true,
            self::Stakeholder     => true,
            self::Tax             => true,
            self::Banking         => true,
            self::Insurance       => true,
            self::Medical         => true,
            self::Legal           => true,
            self::Consultant      => true,
            self::ServiceProvider => true,
            self::LocalSoftware   => true,
            self::Employee        => true,
            self::OwnOrganisation => true,
            self::OwnProduct      => true,
            self::Company         => true,
            self::Association     => true,
            self::Partner         => true,
            self::Family          => true,
            self::Friends         => true,
            self::PrivateOther    => true,
            self::Landlord        => true,

            default               => '* unknown type: ' . $this->value . ' *'
        };
    }

    public function usesServices(): ?string
    {
        return match ($this) {
            self::None            => false,

            self::Customer        => true,
            self::Supplier        => false,
            self::Stakeholder     => true,
            self::Tax             => false,
            self::Banking         => false,
            self::Insurance       => false,
            self::Medical         => false,
            self::Legal           => false,
            self::Consultant      => false,
            self::ServiceProvider => false,
            self::LocalSoftware   => false,
            self::Employee        => true,
            self::OwnOrganisation => true,
            self::OwnProduct      => true,
            self::Company         => true,
            self::Association     => false,
            self::Partner         => true,
            self::Family          => true,
            self::Friends         => true,
            self::PrivateOther    => true,
            self::Landlord        => false,

            default               => '* unknown type: ' . $this->value . ' *'
        };
    }

    public function hasAssets(): ?string
    {
        return match ($this) {
            self::None            => false,

            self::Customer        => false,
            self::Supplier        => false,
            self::Stakeholder     => false,
            self::Tax             => false,
            self::Banking         => false,
            self::Insurance       => false,
            self::Medical         => false,
            self::Legal           => false,
            self::Consultant      => false,
            self::ServiceProvider => false,
            self::LocalSoftware   => false,
            self::Employee        => true,
            self::OwnOrganisation => true,
            self::Company         => true,
            self::Association     => false,
            self::Partner         => true,
            self::Family          => true,
            self::Friends         => true,
            self::PrivateOther    => true,
            self::Landlord        => false,
            self::OwnProduct      => false,

            default               => '* unknown type: ' . $this->value . ' *'
        };
    }

    public function isNaturalPerson(): ?bool
    {
        return match ($this) {
            self::OwnOrganisation => true,
            self::Customer        => true,
            self::Employee        => true,
            self::Partner         => true,
            self::Family          => true,
            self::Friends         => true,
            self::PrivateOther    => true,
            self::Landlord        => true,

            default               => false
        };
    }

    public static function getSupplier(): array
    {
        return array_filter(
            self::cases(),
            fn($case) => $case->providesServices()
        );
    }

    public static function GetTypesForUser(?User $user): array
    {
        $options = [];
        $cases = self::cases();

        if(self::isBusinessUser($user))
        {
            unset($cases[array_search(self::Medical, $cases)]);

            unset($cases[array_search(self::Partner, $cases)]);
            unset($cases[array_search(self::Partner_Current, $cases)]);
            unset($cases[array_search(self::Partner_Former, $cases)]);

            unset($cases[array_search(self::Family, $cases)]);
            unset($cases[array_search(self::Family_Children, $cases)]);
            unset($cases[array_search(self::Family_Parents, $cases)]);
            unset($cases[array_search(self::Family_Sibling, $cases)]);
            unset($cases[array_search(self::Family_Other, $cases)]);

            unset($cases[array_search(self::Friends, $cases)]);
            unset($cases[array_search(self::PrivateOther, $cases)]);
        }

        elseif(self::isPrivateUser($user))
        {
            $cases = [
                self::Partner,
                self::Family,
                self::Friends,
                self::PrivateOther,
                self::Banking,
                self::Insurance,
                self::Medical,
                self::Tax,
                self::Legal,
                self::ServiceProvider,
                self::Consultant,
                self::Landlord,
                self::Stakeholder,
                self::Association,
                self::OwnProduct,
                self::Company,
                self::OwnOrganisation,
            ];
        }

        foreach($cases as $case) {
            $options[$case->value] = $case->getLabel();
        }

        return $options;
    }

    public static function GetSubTypesForType(?OrganisationType $type): array
    {
        $options = [];

        $cases = match ($type)
        {
            self::Partner => [
                self::Partner_Current,
                self::Partner_Former
            ],

            self::Family  => [
                self::Family_Children,
                self::Family_Parents,
                self::Family_Sibling,
                self::Family_Other
            ],

            self::None    => [ // get all subtypes
                self::Partner_Current,
                self::Partner_Former,

                self::Family_Children,
                self::Family_Parents,
                self::Family_Sibling,
                self::Family_Other
            ],

            default       => []
        };

        foreach($cases as $case) {
            $options[$case->value] = $case->getLabel();
        }

        return $options;
    }

    public static function hasSubtypes(?OrganisationType $type): ?string
    {
        return count(self::GetSubTypesForType($type)) > 0;
    }

    public static function isPrivateUser(?User $user = null): bool
    {
        if( ! $user)
        {
            $user = Auth::user();
        }

        return $user->userType->key == self::UserTypePrivate;
    }

    public static function isBusinessUser(?User $user = null): bool
    {
        if( ! $user)
        {
            $user = Auth::user();
        }

        return $user->userType->key == self::UserTypeBusiness;
    }
}