<?php

namespace Netzperfekt\SaasCodoc\Filament;

use Carbon\Carbon;
use Filament\Actions;
use Filament\Actions\MountableAction;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Toggle;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ListRecords;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Actions\BulkAction;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Netzperfekt\SaasCodoc\Enums\OrganisationType;
use Netzperfekt\SaasCodoc\Models\Organisation;
use Netzperfekt\SaasCodoc\Services\ExportService;

class ExportForm extends ListRecords
{
    public static function getExportAction($isBulkAction = false): MountableAction
    {
        if($isBulkAction)
        {
            $action = BulkAction::make('export');
        }
        else {
            $action = Actions\Action::make('export');
        }

        return $action
            ->form(self::getExportForm())
            ->requiresConfirmation()
            ->color(Color::Teal)
            ->modalIcon('heroicon-o-arrow-down-tray')
            ->modalDescription(fn (Collection $records) =>
                    $isBulkAction
                    ? trans_choice('export_documentation_bulk_now', $records->count(), ['count' => $records->count()])
                    : __('export_documentation_now')
            )
            ->modalSubmitActionLabel(__('export'))
            ->modalWidth(MaxWidth::FitContent)
            ->extraModalWindowAttributes(['class' => 'text-3xl'])

            ->action(function (Collection $records, $data, $action) use ($isBulkAction)
            {
                if( ! Organisation::getOwnCompany())
                {
                    Notification::make()
                        ->title(__('codoc-notification-no-organisation'))
                        ->danger()
                        ->send();

                    $action->cancel();
                }

                return response()->streamDownload(function () use ($data, $records, $isBulkAction)
                {
                    $params = [
                        'mydata'   => $data['mydata'] ?? false,
                        'used'     => $data['used'] ?? false,
                        'provided' => $data['provided'] ?? false,
                        'assets'   => $data['assets'] ?? false,
                        'history'  => $data['history'] ?? false,
                    ];

                    if($isBulkAction)
                    {
                        $organisations = $records
                            // TODO sinnvoll? ->where('type', '!=', OrganisationType::OwnOrganisation)
                            ->sortBy('title', SORT_NATURAL|SORT_FLAG_CASE)
                            ->groupBy('type');
                    }
                    else
                    {
                        $organisations = Organisation
                            ::onlyMyRecords()
                            // TODO sinnvoll? ->withoutOwnCompany()
                            ->get()
                            ->sortBy('title', SORT_NATURAL|SORT_FLAG_CASE)
                            ->groupBy('type');
                    }

                    echo app(ExportService::class)
                        ->exportOrganisations($organisations, $params)
                        ->toResponse(request());
                }, self::getExportFileName());
            });
    }

    public static function getExportForm()
    {
        return [
            Section::make('')
                ->columns(3)
                ->schema([
                    Toggle::make('mydata')
                        ->label(
                            OrganisationType::isPrivateUser(Auth()->user()) ?
                                __('codoc-with-mydata') :
                                __('codoc-with-myorganisation')
                        )
                        ->default(true),

                    Toggle::make('used')
                        ->label(__('codoc-with-used'))
                        ->default(true),

                    Toggle::make('provided')
                        ->label(__('codoc-with-provided'))
                        ->default(true),

                    Toggle::make('history')
                        ->label(__('codoc-with-history'))
                        ->default(true),

                    Toggle::make('assets')
                        ->label(__('codoc-with-assets')),

                    Toggle::make('qrcode')
                        ->label(__('codoc-with-qrcode'))
                ])
        ];
    }

    public static function getExportFileName(string $userName = null): string
    {
        return config('app.name') . ' Export ' .
                ($userName ? $userName : Auth::user()->name) . ' ' .
                Carbon::now()->format('d.m.y') . '.pdf';
    }
}
