<?php

namespace Netzperfekt\SaasCodoc\Filament\Pages;

use Filament\Actions\Action;
use Filament\Forms\Components\Actions;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Pages\Concerns\InteractsWithFormActions;
use Filament\Pages\Page;
use Filament\Support\Exceptions\Halt;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasCodoc\Models\Service;

class Integrations extends Page implements HasForms
{
    use InteractsWithFormActions;

    protected static ?string $navigationIcon = 'fas-arrows-to-eye';
    protected static string $view = 'saas-codoc::filament.integrations';

    public array $data;

    protected function getActions(): array
    {
        return [
        ];
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')
                ->label(__('save'))
                ->submit('data')
        ];
    }

    protected function getFormStatePath(): string
    {
        return 'data';
    }

    protected function getFormSchema(): array
    {
        return [
            Placeholder::make('userinfo')
                ->label(__('userinfo'))
                ->content(function ()
                {
                    $s = '<b>' . auth()->user()->name . '</b>';

                    return new HtmlString($s);
                }),

            Placeholder::make('share_info')
                ->label('')
                ->content(fn () => new HtmlString(__('share_info_description')))
                ->extraAttributes(['class' => 'info-panel']),

            Grid::make('tokens_all')
                ->label(__('tokens'))
                ->columns(20)
                ->extraAttributes(['class' => 'content-panel'])
                ->schema([
                    Toggle::make('share_all')
                        ->label(__('share_all'))
                        ->columnSpanFull()
                        ->hint(fn() => new HtmlString(__('share_all_hint')))
                        ->live()
                        ->default(false),

                    Grid::make('tokens_all_fields')
                        ->label('')
                        ->visible(fn($get) => $get('share_all'))
                        ->columnStart(2)
                        ->schema([
                            Actions::make([
                                Actions\Action::make('generate_token_all')
                                    ->label(__('generate_token_all'))
                                    ->requiresConfirmation()
                                    ->modalDescription(__('generate_token_all_description'))
                                    ->color('info')
                                    ->visible(fn() => count(auth()->user()->getTokens(config('saas-codoc.token_name_all'))) == 0)
                                    ->action(function ()
                                    {
                                        auth()->user()->createToken(config('saas-codoc.token_name_all'));
                                    })
                            ]),
                            Actions::make([
                                Actions\Action::make('revoke_token_all')
                                    ->label(__('revoke_token_all'))
                                    ->requiresConfirmation()
                                    ->modalDescription(__('revoke_token_all_description'))
                                    ->color('danger')
                                    ->visible(fn() => count(auth()->user()->getTokens(config('saas-codoc.token_name_all'))) > 0)
                                    ->action(function ()
                                    {
                                        auth()->user()->tokens()->where('name', config('saas-codoc.token_name_all'))->delete();
                                    })
                            ]),

                            Placeholder::make('tokens_all_info')
                                ->label('')
                                ->visible(fn() => count(auth()->user()->getTokens(config('saas-codoc.token_name_all'))) > 0)
                                ->content(function ()
                                {
                                    $token = auth()->user()->getTokens(config('saas-codoc.token_name_all'))->first();

                                    $url = URL::signedRoute(
                                        'codoc.shared.all', [
                                            'token' => $token->token
                                        ]
                                    );

                                    $s  = '<ul><li>';
                                    $s .= '<span class="share-link"><a href="' . $url . '" target="_blank" class="link-button">' . __('share_link_all') . '</a></span>';
                                    $s .= '<span class="badge">' . $token->token . '</span>';
                                    $s .= '</li></ul>';

                                    return new HtmlString($s);
                                })
                        ])
                ]),

            Grid::make('tokens_all')
                ->label(__('tokens'))
                ->columns(20)
                ->extraAttributes(['class' => 'content-panel'])
                ->schema([
                    Toggle::make('share_projects')
                        ->label(__('share_projects'))
                        ->hint(fn() => new HtmlString(__('share_projects_hint')))
                        ->columnSpanFull()
                        ->live()
                        ->default(false),

                    Grid::make('tokens_project_fields')
                        ->label('')
                        ->visible(fn($get) => $get('share_projects'))
                        ->columnStart(2)
                        ->schema([
                            Actions::make([
                                Actions\Action::make('generate_token_projects')
                                    ->label(__('generate_token_projects'))
                                    ->requiresConfirmation()
                                    ->modalDescription(__('generate_token_projects_description'))
                                    ->color('info')
                                    ->visible(fn() => count(auth()->user()->getTokens(config('saas-codoc.token_name_projects'))) == 0)
                                    ->action(function ()
                                    {
                                        foreach(Service::onlyMyRecords()->projects()->get() as $project)
                                        {
                                            auth()->user()->createToken($this->getProjectToken($project));
                                        }
                                    })
                            ]),
                            Actions::make([
                                Actions\Action::make('revoke_token_projects')
                                    ->label(__('revoke_token_projects'))
                                    ->requiresConfirmation()
                                    ->modalDescription(__('revoke_token_projects_description'))
                                    ->color('danger')
                                    ->visible(fn() => count(auth()->user()->getTokens(config('saas-codoc.token_name_projects'))) > 0)
                                    ->action(function ()
                                    {
                                        foreach(Service::onlyMyRecords()->projects()->get() as $project)
                                        {
                                            auth()->user()->tokens()->where('name', $this->getProjectToken($project))->delete();
                                        }

                                    })
                            ]),

                            Placeholder::make('tokens_projects_info')
                                ->label('')
                                ->visible(fn() => count(auth()->user()->tokens) > 0)
                                ->content(function ()
                                {
                                    $projects = Service::onlyMyRecords()->projects()->get();

                                    $s = '<ul>';
                                    foreach ($projects as $project)
                                    {
                                        $token = auth()->user()->getTokens($this->getProjectToken($project))->first();

                                        if($token) {
                                            $url = URL::signedRoute(
                                                'codoc.shared.project', [
                                                    'token' => $token->token,
                                                    'projectId' => $project->id
                                                ]
                                            );

                                            $s .= '<li>';
                                            $s .= '<span class="share-link"><a href="' . $url . '" target="_blank" class="link-button">' . $project->title . '</a></span>';
                                            $s .= '<span class="badge">' . $token->token . '</span>';
                                            $s .= '</li>';
                                        }
                                    }
                                    $s .= '</ul>';

                                    return new HtmlString($s);
                                })
                        ])
                ]),

            Placeholder::make('bizguard_info')
                ->label('')
                ->content(fn () => new HtmlString(__('share_bizguard_description')))
                ->extraAttributes(['class' => 'action-panel mb-8']),
        ];
    }

    public function mount(): void
    {
        $config = auth()->user()->config ?? [];
        $formValues = [
            'share_all'      => $config['share_all'] ?? false,
            'share_projects' => $config['share_projects'] ?? false
        ];

        $this->form->fill($formValues);
    }

    public function submit(): void
    {
        try
        {
            $config = auth()->user()->config ?? [];
            $config['share_all'] = $this->data['share_all'];
            $config['share_projects'] = $this->data['share_projects'];

            auth()->user()->config = $config;
            auth()->user()->save();
        }
        catch (Halt $exception)
        {
            return;
        }

        Notification::make()
            ->title(__('saved'))
            ->success()
            ->send();
    }

    public static function getNavigationGroup(): ?string
    {
        return __(config('saas-base.settings_group'));
    }

    public static function getNavigationLabel(): string
    {
        return __('shares');
    }

    public function getTitle(): string
    {
        return __('shares');
    }

    private function getProjectToken(Service $project): string
    {
        return config('saas-codoc.token_name_projects') . $project->id;
    }
}
