<?php

namespace Netzperfekt\SaasCodoc\Filament\Resources\AssetResource;

use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Guava\FilamentKnowledgeBase\Contracts\HasKnowledgeBase;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Netzperfekt\SaasCodoc\Enums\AssetStatus;
use Netzperfekt\SaasCodoc\Enums\AssetType;
use Netzperfekt\SaasCodoc\Filament\Components\Comments;
use Netzperfekt\SaasCodoc\Filament\Components\CommentsColumn;
use Netzperfekt\SaasCodoc\Filament\Resources\AssetResource\Pages\CreateAsset;
use Netzperfekt\SaasCodoc\Filament\Resources\AssetResource\Pages\EditAsset;
use Netzperfekt\SaasCodoc\Filament\Resources\AssetResource\Pages\ListAssets;
use Netzperfekt\SaasCodoc\Models\Asset;
use Netzperfekt\SaasCodoc\Models\Organisation;

class AssetResource extends Resource implements HasKnowledgeBase
{
    protected static ?string $model = Asset::class;

    protected static ?string $navigationIcon = 'fas-laptop';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = -1;

    public static function getNavigationGroup(): ?string
    {
        return __('codoc-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return __('codoc-asset');
    }

    public static function getPluralModelLabel(): string
    {
        return __('codoc-assets');
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['title', 'key', 'serial_number', 'inventory_number'];
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Typ' => $record->type->getLabel()
        ];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema(self::getFormSchema());
    }

    public static function getFormSchema(int $organisationId = 0): array
    {
        return [
            Split::make([
                Section::make(__('codoc-asset'))
                    ->schema([
                        Grid::make()
                            ->columns(5)
                            ->schema([
                                TextInput::make('key')
                                    ->label(__('codoc-key'))
                                    ->required()
                                    ->maxLength(25)
                                    ->autofocus(),

                                TextInput::make('title')
                                    ->label(__('codoc-title'))
                                    ->columnSpan(3)
                                    ->required()
                                    ->maxLength(255),

                                DatePicker::make('buy_date')
                                    ->label(__('codoc-buy-date'))
                                    ->native(false)
                                    ->displayFormat('d.m.Y')
                                    ->required(),

                                TextInput::make('manufacturer')
                                    ->label(__('codoc-manufacturer'))
                                    ->required()
                                    ->maxLength(255),

                                TextInput::make('model')
                                    ->label(__('codoc-model'))
                                    ->maxLength(255),

                                TextInput::make('serial_number')
                                    ->label(__('codoc-serial-number'))
                                    ->maxLength(255),

                                TextInput::make('inventory_number')
                                    ->label(__('codoc-inventory-number'))
                                    ->maxLength(255),

                                TextInput::make('location')
                                    ->label(__('codoc-location'))
                                    ->maxLength(255),
                            ]),

                        Textarea::make('description')
                            ->label(__('codoc-description'))
                            ->rows(2)->autosize(),
                    ])
                    ->collapsible()
                    ->persistCollapsed()
                    ->grow(),

                Section::make([
                    Select::make('type')
                        ->label(__('codoc-type'))
                        ->options(AssetType::class)
                        ->live()
                        ->required(),

                    Select::make('status')
                        ->label(__('codoc-status'))
                        ->options(AssetStatus::class)
                        ->default(AssetStatus::Active)
                        ->required(),

                    TagsInput::make('tags')
                        ->label(__('codoc-tags'))
                        ->splitKeys([','])
                        ->maxWidth(MaxWidth::ExtraSmall),

                    Select::make('organisation_id')
                        ->label(__('codoc-organisation'))
                        ->relationship(
                            name: 'organisation',
                            titleAttribute: 'title',
                            modifyQueryUsing: fn (Builder $query) => $query->onlyMyRecords()
                        )
                        ->searchable()
                        ->preload()
                        ->default(fn() => $organisationId ? $organisationId : Organisation::OwnCompany()?->get()?->first()?->id),

                ])->grow(false)
            ])->from('xl'),

            Section::make(__('codoc-comments'))
                ->schema([
                    Comments::make('comments')
                        ->label('')
                        ->user(Auth::user())
                        ->editableOnlyOwnValues()
                ])
                ->compact()
                ->collapsible()
                ->persistCollapsed()
        ];
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultPaginationPageOption(25)
            ->groups([
                Tables\Grouping\Group::make('type')
                    ->label(__('codoc-type'))
                    ->collapsible()
            ])

            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('codoc-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('key')
                    ->label(__('codoc-key'))
                    ->searchable(),

                Tables\Columns\TextColumn::make('tags')
                    ->tags(),

                CommentsColumn::make('comments')
                    ->tooltip(__('codoc-comments'))
                    ->showCount(),

                Tables\Columns\TextColumn::make('organisation.title')
                    ->label(__('codoc-organisation'))
                    ->badge(),

                Tables\Columns\TextColumn::make('type')
                    ->label(__('codoc-type'))
                    ->badge()
                    ->color(Color::Green),

                Tables\Columns\TextColumn::make('status')
                    ->label(__('codoc-status'))
                    ->badge()
            ])

            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('type')
                    ->label(__('codoc-type'))
                    ->options(AssetType::class),

                Tables\Filters\SelectFilter::make('status')
                    ->label(__('codoc-status'))
                    ->options(AssetStatus::class),

                Tables\Filters\SelectFilter::make('organisation')
                    ->label(__('codoc-organisation'))
                    ->relationship('organisation', 'title')
            ])

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),
                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])

            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ])

            ->defaultSort('title');
    }

    public static function getRelations(): array
    {
        return [
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListAssets::route('/'),
            'create' => CreateAsset::route('/create'),
            'edit' => EditAsset::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    public static function getDocumentation(): array
    {
        return [
            'assets'
        ];
    }
}
