<?php

namespace Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource;

use Filament\Facades\Filament;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconPosition;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Guava\FilamentKnowledgeBase\Contracts\HasKnowledgeBase;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Netzperfekt\SaasBase\SaasBase;
use Netzperfekt\SaasCodoc\Enums\OrganisationStatus;
use Netzperfekt\SaasCodoc\Enums\OrganisationType;
use Netzperfekt\SaasCodoc\Filament\Components\Comments;
use Netzperfekt\SaasCodoc\Filament\Components\CommentsColumn;
use Netzperfekt\SaasCodoc\Filament\ExportForm;
use Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\Pages\CreateOrganisationPage;
use Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\Pages\EditOrganisationPage;
use Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\Pages\ListOrganisationPages;
use Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\RelationManagers\OrganisationAssetsRelationManager;
use Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\RelationManagers\OrganisationProvidesServicesRelationManager;
use Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\RelationManagers\OrganisationUsesServicesRelationManager;
use Netzperfekt\SaasCodoc\Models\Organisation;

class OrganisationPageResource extends Resource implements HasKnowledgeBase
{
    protected static ?string $model = Organisation::class;

    protected static ?string $navigationIcon = 'heroicon-o-building-office';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = -2;

    public static function getNavigationGroup(): ?string
    {
        return __('codoc-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return OrganisationType::isPrivateUser() ? __('codoc-organisation-contact') : __('codoc-organisation');
    }

    public static function getPluralModelLabel(): string
    {
        return OrganisationType::isPrivateUser() ? __('codoc-organisations-contacts') : __('codoc-organisations');
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['title', 'company1', 'company2'];
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Typ' => $record->type->getLabel()
        ];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Split::make([
                    Section::make(__('codoc-maindata'))
                    ->schema([
                        Section::make('')
                            ->compact()
                            ->columns(3)
                            ->schema([
                                TextInput::make('title')
                                    ->label(__('codoc-title'))
                                    ->extraAttributes(['class' => 'font-semibold'])
                                    ->required()
                                    ->maxLength(255)
                                    ->autofocus()
                                    ->default(fn() => Organisation::onlyMyRecords()->count() == 0 ? Auth()->user()->name : ''),

                                TextInput::make('company1')
                                    ->label(__('codoc-company1'))
                                    ->required(fn($get) => $get('usertype') == OrganisationType::UserTypeBusiness)
                                    ->maxLength(255),

                                TextInput::make('company2')
                                    ->label(__('codoc-company2'))
                                    ->maxLength(255),

                                Grid::make()
                                    ->columns(4)
                                    ->schema([
                                    TextInput::make('street')
                                        ->label(__('codoc-street'))
                                        ->required()
                                        ->maxLength(255),

                                    TextInput::make('zip')
                                        ->label(__('codoc-zip'))
                                        ->required()
                                        ->maxLength(255),

                                    TextInput::make('city')
                                        ->label(__('codoc-city'))
                                        ->required()
                                        ->maxLength(255),

                                    Select::make('country')
                                        ->label(__('codoc-country'))
                                        ->options(SaasBase::getCountryOptions())
                                        ->default(strtoupper(app()->getLocale()))
                                        ->required()
                                ]),

                                TextInput::make('phone')
                                    ->label(__('codoc-phone'))
                                    ->required()
                                    ->maxLength(255),

                                TextInput::make('email')
                                    ->label(__('codoc-email'))
                                    ->required()
                                    ->email()->nullable()
                                    ->maxLength(255)
                                    ->default(fn() => Organisation::onlyMyRecords()->count() == 0 ? Auth()->user()->email : ''),

                                TextInput::make('homepage')
                                    ->label(__('codoc-homepage'))
                                    ->url()
                                    ->maxLength(255)
                                    ->live()
                                    ->suffixAction(Action::make('open-homepage-url')
                                        ->icon('fas-link')
                                        ->url(fn($get) => $get('homepage'))
                                        ->disabled(fn($get) => $get('homepage') == '')
                                        ->openUrlInNewTab()),

                                RichEditor::make('contact')
                                    ->label(OrganisationType::isPrivateUser() ? __('codoc-contact-private') : __('codoc-contact'))
                                    ->columnSpanFull()
                                    ->maxLength(1024)
                                    ->defaultRichEditor()
                            ]),

                        Section::make(__('codoc-additional'))
                            ->schema([
                                Section::make('')
                                    ->compact()
                                    ->columns(3)
                                    ->schema(fn($get) => OrganisationAdditionalFields::getAdditionalFields(OrganisationType::from($get('type'))))
                            ])->compact()
                              ->visible(function($get)
                              {
                                  return count(OrganisationAdditionalFields::getAdditionalFields(OrganisationType::from($get('type')))) > 0;
                              })
                            ->live()

                    ])->compact()
                        ->collapsible()
                        ->persistCollapsed()
                        ->grow(),

                    Section::make([
                        Select::make('type')
                            ->label(__('codoc-type'))
                            ->required()
                            ->options(OrganisationType::GetTypesForUser(auth()->user()))
                            ->live()
                            ->default(fn()
                                => Organisation::onlyMyRecords()->count() == 0 ?
                                        OrganisationType::OwnOrganisation->value :
                                        OrganisationType::None->value)
                            ->disabled(fn()
                                => Organisation::onlyMyRecords()->count() == 0)
                            ->dehydrated(),

                        Select::make('subtype')
                            ->label(__('codoc-subtype'))
                            ->options(fn($get) => OrganisationType::GetSubTypesForType(OrganisationType::from($get('type'))))
                            ->visible(fn($get) => OrganisationType::hasSubtypes(OrganisationType::from($get('type'))))
                            ->live(),

                        Select::make('status')
                            ->label(__('codoc-status'))
                            ->required()
                            ->options(OrganisationStatus::class)
                            ->default(OrganisationStatus::Active)
                            ->live(),

                        TagsInput::make('authorization_tags')
                            ->label(__('codoc-authorization'))
                            ->hintIcon(
                                'heroicon-m-question-mark-circle',
                                tooltip: OrganisationType::isPrivateUser() ? __('codoc-authorization-help-private') : __('codoc-authorization-help')
                            )
                            ->splitKeys([','])
                            ->maxWidth(MaxWidth::ExtraSmall)
                            ->grow(false),

                        Select::make('assigned_organisations')
                            ->label(__('codoc-assigned-organisations'))
                            ->hintIcon(
                                'heroicon-m-question-mark-circle',
                                tooltip: __('codoc-assigned-organisations-help')
                            )
                            ->options(Organisation::onlyMyRecords()->orderBy('title')->get()->pluck('title', 'id'))
                            ->searchable()
                            ->multiple(),

                        FileUpload::make('documents')
                            ->label(__('codoc-documents'))
                            ->multiple()
                            ->directory(fn($record) => 'doc/t' . Filament::getTenant()->id . '_o' . $record->id)
                            ->disk(config('saas-codoc.attachments_disk'))
                            ->maxSize(config('saas-codoc.attachments_max_size'))
                            ->visibility('private')
                            ->columnSpan(1)
                            ->downloadable(true)
                            ->openable(true)
                            ->moveFiles()
                            ->preserveFilenames()

                            ->disabled(fn($operation) => $operation == 'create')
                            ->hintIcon(
                                fn($operation) => $operation == 'create' ? 'heroicon-m-question-mark-circle' : '',
                                fn($operation) => $operation == 'create' ? __('codoc-msg-save-first') : ''
                            )
                    ])->grow(false)
                ])->from('xl'),

                Section::make(__('codoc-comments'))
                    ->schema([
                        Comments::make('comments')
                            ->label('')
                            ->user(Auth::user())
                            ->editableOnlyOwnValues(),
                    ])
                        ->compact()
                        ->collapsible()
                        ->persistCollapsed(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultPaginationPageOption(25)
            ->groups([
                Tables\Grouping\Group::make('type')
                    ->label(__('codoc-type'))
                    ->collapsible(),

                Tables\Grouping\Group::make('subtype')
                    ->label(__('codoc-subtype'))
                    ->collapsible()
            ])

            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('codoc-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable()
                    ->tooltip(function ($record)
                    {
                        if($record->isOwnCompany()) {
                            return OrganisationType::isPrivateUser() ? __('codoc-myself') : __('codoc-own-company');
                        }
                        elseif($record->isOwnProduct()) {
                            return __('codoc-own-product');
                        }

                        return '';
                    })
                    ->icon(function ($record)
                    {
                        if($record->isOwnCompany() || $record->isOwnProduct()) {
                            return $record->type->getIcon();
                        }

                        return '';
                    })
                    ->iconColor('warning')
                    ->iconPosition(IconPosition::After),


                Tables\Columns\TextColumn::make('company1')
                    ->label(__('codoc-company1'))
                    ->limit(25)
                    ->searchable(),

                Tables\Columns\TextColumn::make('services_providing_count')
                    ->counts('servicesProviding')
                    ->formatStateUsing(fn($state) => $state > 0 ? $state : '')
                    ->label(__('codoc-services-providing-count'))
                    ->tooltip(__('codoc-organisations-providing'))
                    ->badge()
                    ->color(Color::Yellow)
                    ->modalServicesProviding(),

                Tables\Columns\TextColumn::make('services_using_count')
                    ->counts('servicesUsing')
                    ->formatStateUsing(fn($state) => $state > 0 ? $state : '')
                    ->label(__('codoc-services-using-count'))
                    ->tooltip(__('codoc-organisations-using'))
                    ->badge()
                    ->color(Color::Yellow)
                    ->modalServicesUsing(),

                CommentsColumn::make('comments')
                    ->tooltip(__('codoc-comments'))
                    ->showCount(),

                Tables\Columns\TextColumn::make('type')
                    ->label(__('codoc-type'))
                    ->tooltip(fn($record) => OrganisationType::hasSubtypes($record->type) ? $record->subtype?->getLabel() ?? '' : '')
                    ->badge()
                    ->color(Color::Green),

                Tables\Columns\TextColumn::make('status')
                    ->label(__('codoc-status'))
                    ->badge()
            ])
            
            ->recordClasses(fn($record) => $record->type == OrganisationType::OwnOrganisation ? 'bg-blue-50 dark:bg-gray-600' : null)

            ->filters([
                Tables\Filters\SelectFilter::make('type')
                    ->label(__('codoc-type'))
                    ->options(OrganisationType::GetTypesForUser(auth()->user())),

                // TODO abhängiger filter wäre schön
                // https://v2.filamentphp.com/tricks/dependent-select-filters

                Tables\Filters\SelectFilter::make('subtype')
                    ->label(__('codoc-subtype'))
                    ->options(OrganisationType::GetSubTypesForType(OrganisationType::None)),

                Tables\Filters\SelectFilter::make('status')
                    ->label(__('codoc-status'))
                    ->options(OrganisationStatus::class),

                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('without_associations')
                    ->indicator('without_associations')
                    ->indicateUsing(fn(array $state) => $state['without_associations'] ? __('codoc-without-associations') : null)
                    ->form([
                        Toggle::make('without_associations')->label(__('codoc-without-associations'))

                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['without_associations'],
                                fn (Builder $query, $date): Builder =>
                                $query->doesntHave('servicesProviding')
                                      ->doesntHave('servicesUsing')
                                      ->withoutTrashed()
                            );
                    })

            ])

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),
                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])

            ->bulkActions([
                ExportForm::getExportAction(true),

                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ])
            
            ->defaultSort(function (Builder $query): Builder {
                return $query
                    ->orderByRaw('IF(type = ' . OrganisationType::OwnOrganisation->value .
                                     ' || type = ' . OrganisationType::OwnProduct->value . ', -1, title)');
            });
    }

    public static function getRelations(): array
    {
        return [
            'servicesproviding' => OrganisationProvidesServicesRelationManager::class,
            'servicesusing'     => OrganisationUsesServicesRelationManager::class,
            'assets'            => OrganisationAssetsRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListOrganisationPages::route('/'),
            'create' => CreateOrganisationPage::route('/create'),
            'edit' => EditOrganisationPage::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    public static function getDocumentation(): array
    {
        return [
            'organisations'
        ];
    }
}
