<?php

namespace Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\RelationManagers;

use Filament\Facades\Filament;
use Filament\Forms;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\KeyValue;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconPosition;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\On;
use Netzperfekt\SaasCodoc\Enums\ServiceStatus;
use Netzperfekt\SaasCodoc\Enums\ServiceType;
use Netzperfekt\SaasCodoc\Filament\Components\Comments;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\ServicePageResource;
use Netzperfekt\SaasCodoc\Models\Organisation;
use Netzperfekt\SaasCodoc\Models\Service;
use Netzperfekt\SaasCodoc\Models\ServicesComponents;

class OrganisationProvidesServicesRelationManager extends RelationManager
{
    protected static string $relationship = 'servicesProviding';
    protected static ?string $inverseRelationship = 'organisationsProviding';
    protected static bool $isLazy = false;

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return __('codoc-services-providing');
    }

    public static function getRecordLabel(): string
    {
        return __('codoc-service');
    }

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return $ownerRecord->type->providesServices();
    }

    public function mount(): void
    {
        parent::mount();

        $this->dispatch('parse-deeplink');
    }

    public function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Forms\Components\Grid::make(6)
                    ->schema([
                        Forms\Components\TextInput::make('service_title')
                            ->label(__('codoc-title'))
                            ->columnSpan(3)
                            ->required(),

                        Select::make('type')
                            ->label(__('codoc-type'))
                            ->options(ServiceType::class)
                            ->disabled(),

                        Select::make('service_status')
                            ->label(__('codoc-status'))
                            ->options(fn($get) => ServiceStatus::getOptionsForType(ServiceType::from($get('type'))))
                            ->default(ServiceStatus::Active)
                            ->required(),

                        Toggle::make('is_favorite')
                            ->inline(false)
                            ->label(__('codoc-is-favorite'))
                            ->onIcon('fas-star')
                            ->offIcon('far-star')
                    ]),

                Forms\Components\RichEditor::make('service_description')
                    ->label(__('codoc-description'))
                    ->defaultRichEditor(),

                Forms\Components\Grid::make(5)
                    ->schema([
                        Forms\Components\TextInput::make('service_url')
                            ->url()
                            ->live()
                            ->label(__('codoc-service-url'))
                            ->columnSpan(2)
                            ->suffixAction(Forms\Components\Actions\Action::make('open-service-url')
                                ->icon('fas-link')
                                ->url(fn($get) => $get('service_url'))
                                ->disabled(fn($get) => $get('service_url') == '')
                                ->openUrlInNewTab()),

                        Toggle::make('service_needs_dpa')
                            ->live()
                            ->label(__('codoc-needs_dpa')),

                        FileUpload::make('service_documents')
                            ->label(__('codoc-documents'))
                            ->multiple()
                            ->directory(fn($record) => 'doc/t' . Filament::getTenant()->id . '_o' . $record->id)
                            ->disk(config('saas-codoc.attachments_disk'))
                            ->maxSize(config('saas-codoc.attachments_max_size'))
                            ->visibility('private')
                            ->columnSpan(2)
                            ->downloadable(true)
                            ->openable(true)
                            ->moveFiles()
                            ->preserveFilenames()
                    ]),

                KeyValue::make('service_additional')
                    ->label(__('codoc-additional-fields'))
                    ->editableKeys(false)
                    ->addable(false)
                    ->deletable(false)
                    ->visible(fn($get) => ! empty($get('service_additional'))),

                Comments::make('service_comments')
                    ->label(__('codoc-comments'))
                    ->user(Auth::user()),
            ]);
    }

    public function table(Table $table): Table
    {
        $editAction = Tables\Actions\EditAction::make('edit')
            ->label('')
            ->tooltip('Bearbeiten')
            ->iconSize(IconSize::Medium)
            ->modalWidth(MaxWidth::FiveExtraLarge)
            ->mutateRecordDataUsing(function($data)
            {
                if(empty($data['service_title']))
                {
                    $data['service_title'] = $data['title'];
                }

                if(empty($data['service_description']))
                {
                    $data['service_description'] = $data['description'];
                }

                if(empty($data['service_url']))
                {
                    $data['service_url'] = $this->ownerRecord->homepage;
                }

                if(empty($data['service_additional']))
                {
                    $data['service_additional'] = $data['additional_template'] ?? [];
                }

                return $data;
            });

        return $table
            ->recordTitleAttribute('title')
            ->allowDuplicates()

            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('codoc-service'))
                    ->weight(FontWeight::SemiBold)
                    ->icon(fn($record) => count($record?->service_documents ?? []) > 0 ? 'far-file' : '')
                    ->iconPosition(IconPosition::After)
                    ->tooltip(fn($record) => $record->service_needs_dpa ? __('codoc-dpa-present') : '')
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('components_count')
                    ->label(__('codoc-services-components'))
                    ->badge()
                    ->getStateUsing(function ($record)
                    {
                        $count = count(ServicesComponents::where('service_id', $record->service_id)->get());
                        return $count ? $count : '';
                    })
                    ->tooltip(function ($record)
                    {
                        $components = ServicesComponents::where('service_id', $record->service_id)->get();
                        return $components->pluck('service.title')->implode(' | ');
                    }),

                Tables\Columns\TextColumn::make('organisations_using_count')
                    ->label(__('codoc-organisations-using'))
                    ->countWithMissingAlert(
                        fn($record) => $record->service->organisationsUsing->count(),
                        fn($record) => __('codoc-organisations-using')
                    )
                    ->modalOrganisationsUsing(),

                Tables\Columns\TextColumn::make('service_title')
                    ->label(__('codoc-title'))
                    ->icon('fas-circle-info')
                    ->iconPosition(IconPosition::After)
                    ->tooltip(fn($record) => $record->type->getLabel())
                    ->sortable()
                    ->searchable(),

                Tables\Columns\IconColumn::make('service_url')
                    ->openUrl(),

                Tables\Columns\TextColumn::make('service_status')
                    ->label(__('codoc-status'))
                    ->badge()
            ])

            ->filters([
                Tables\Filters\Filter::make('dpa_not_present')
                    ->label(__('codoc-dpa-not-present'))
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->hasNoDpa()),
            ])

            ->headerActions([
                Tables\Actions\Action::make('create')
                    ->label(__('codoc-create'))
                    ->keyBindings(['command+e', 'ctrl+e'])
                    ->modal()
                    ->model(Service::class)
                    ->form(ServicePageResource::getFormFields(true))
                    ->action(function ($livewire, array $data) use($editAction)
                    {
                        $service = Service::create($data);
                        $livewire->ownerRecord->servicesProviding()->attach($service);

                        if($data['assignToOwnOrganisation'] === true)
                        {
                            Organisation::getOwnCompany()->servicesUsing()->attach(
                                $service,
                                [
                                    'organisation_provided_id' => $livewire->ownerRecord->id,
                                    'service_title'            => $data['title'],
                                    'service_description'      => $data['description'],
                                    'service_url'              => $livewire->ownerRecord->homepage,
                                ]
                            );
                        }

                        // TODO call edit action, but how?

                    }),

                Tables\Actions\AttachAction::make()
                    ->color(Color::Yellow)
                    ->label(__('codoc-service-assign'))
                    ->recordSelect(
                        fn (Select $select) =>
                            $select->placeholder('Service auswählen')
                                   ->autofocus()
                    )
                    ->recordSelectOptionsQuery(fn(Builder $query) => $query->onlyMyRecords()->active()->orderBy('title'))
                    ->preloadRecordSelect()
            ])

            ->actions([
                $editAction,

                Tables\Actions\Action::make('edit_service')
                    ->label('')
                    ->icon('heroicon-o-arrow-right-start-on-rectangle')
                    ->color(Color::Orange)
                    ->iconSize(IconSize::Medium)
                    ->tooltip(__('codoc-edit-service'))
                    ->url(fn (Service $record): string => ServicePageResource::getUrl('edit', ['record' => $record->service_id])),

                Tables\Actions\DetachAction::make()
                    ->label('')
                    ->tooltip('Trennen')
                    ->iconSize(IconSize::Medium)
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DetachBulkAction::make()
                ]),
            ])

            ->defaultSort('sort_order', 'asc')

            ->reorderable('sort_order', true)
            ->reorderRecordsTriggerAction(
                fn (Tables\Actions\Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? __('codoc-sort-save') : __('codoc-sort'))
                    ->color($isReordering ? 'warning' : 'gray')
            );
    }

    #[On('deeplink-servicesproviding')]
    public function executeDeeplink(string $action, string $id)
    {
        $this->replaceMountedTableAction($action, $id);
    }
}
