<?php

namespace Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\RelationManagers;

use Filament\Facades\Filament;
use Filament\Forms;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\KeyValue;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconPosition;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Netzperfekt\SaasCodoc\Enums\OrganisationType;
use Netzperfekt\SaasCodoc\Enums\ServiceStatus;
use Netzperfekt\SaasCodoc\Enums\ServiceType;
use Netzperfekt\SaasCodoc\Filament\Components\Comments;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\ServicePageResource;
use Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\OrganisationPageResource;
use Netzperfekt\SaasCodoc\Models\Organisation;
use Netzperfekt\SaasCodoc\Models\OrganisationProvidesServices;
use Netzperfekt\SaasCodoc\Models\Service;
use Netzperfekt\SaasCodoc\Models\ServicesComponents;
use Livewire\Attributes\On;

class OrganisationUsesServicesRelationManager extends RelationManager
{
    protected static string $relationship = 'servicesUsing';
    protected static ?string $inverseRelationship = 'organisationsUsing';
    protected static bool $isLazy = false;

    public static function getRecordLabel(): string
    {
        return __('codoc-service');
    }

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return __('codoc-services-using');
    }

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return $ownerRecord->type->usesServices();
    }

    public function mount(): void
    {
        parent::mount();

        $this->dispatch('parse-deeplink');
    }

    public function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Forms\Components\Grid::make(6)
                    ->schema([
                        Forms\Components\TextInput::make('service_title')
                            ->label(__('codoc-title'))
                            ->columnSpan(3)
                            ->required(),

                        Select::make('type')
                            ->label(__('codoc-type'))
                            ->options(ServiceType::class)
                            ->disabled(),

                        Select::make('service_status')
                            ->label(__('codoc-status'))
                            ->options(fn($get) => ServiceStatus::getOptionsForType(ServiceType::from($get('type'))))
                            ->required(),

                        Toggle::make('is_favorite')
                            ->inline(false)
                            ->label(__('codoc-is-favorite'))
                            ->onIcon('fas-star')
                            ->offIcon('far-star')
                    ]),

                Forms\Components\RichEditor::make('service_description')
                    ->label(__('codoc-description'))
                    ->defaultRichEditor(),

                Forms\Components\Select::make('organisation_provided_id')
                    ->label(__('codoc-supplier'))
                    ->options(Organisation::supplierOrOwnCompany()->orderBy('title')->get()->pluck('title', 'id'))
                    ->suffixAction(Action::make('show-provider')
                        ->icon('heroicon-o-building-office')
                        ->tooltip(__('codoc-edit-organisation'))
                        ->url(fn($record) => OrganisationPageResource::getUrl('edit', ['record' => $record->organisation_provided_id]))
                        ->openUrlInNewTab())
                        ->searchable(),

                Forms\Components\Grid::make(5)
                    ->schema([
                        Forms\Components\TextInput::make('service_url')
                            ->url()
                            ->label(__('codoc-service-url'))
                            ->columnSpan(2)
                            ->live()
                            ->suffixAction(Action::make('open-service-url')
                                ->icon('fas-link')
                                ->url(fn($get) => $get('service_url'))
                                ->disabled(fn($get) => $get('service_url') == '')
                                ->openUrlInNewTab()),

                        Toggle::make('service_needs_dpa')
                            ->visible(fn($record) => OrganisationType::isPrivateUser(Auth()->user()) ? false : true)
                            ->live()
                            ->label(__('codoc-needs_dpa')),

                        FileUpload::make('service_documents')
                            ->label(__('codoc-documents'))
                            ->multiple()
                            ->directory(fn($record) => 'doc/t' . Filament::getTenant()->id . '_o' . $record->id)
                            ->disk(config('saas-codoc.attachments_disk'))
                            ->maxSize(config('saas-codoc.attachments_max_size'))
                            ->visibility('private')
                            ->columnSpan(2)
                            ->downloadable(true)
                            ->openable(true)
                            ->moveFiles()
                            ->preserveFilenames()
                    ]),

                KeyValue::make('service_additional')
                    ->label(__('codoc-additional-fields'))
                    ->editableKeys(false)
                    ->addable(false)
                    ->deletable(false)
                    ->visible(fn($get) => ! empty($get('service_additional')) && count($get('service_additional')) > 0),

                Comments::make('service_comments')
                    ->label(__('codoc-comments'))
                    ->user(Auth::user())
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('title')
            ->allowDuplicates()

            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('codoc-service'))
                    ->weight(FontWeight::SemiBold)
                    ->icon(fn($record) => count($record?->service_documents ?? []) > 0 ? 'far-file' : '')
                    ->iconPosition(IconPosition::After)
                    ->tooltip(fn($record) => $record->service_needs_dpa ? __('codoc-dpa-present') : '')
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('components_count')
                    ->label(__('codoc-services-components'))
                    ->badge()
                    ->getStateUsing(function ($record)
                    {
                        $count = count(ServicesComponents::where('service_id', $record->service_id)->get());
                        return $count ? $count : '';
                    })
                    ->tooltip(function ($record)
                    {
                        $components = ServicesComponents::where('service_id', $record->service_id)->get();
                        return $components->pluck('service.title')->implode(' | ');
                    }),

                Tables\Columns\TextColumn::make('service_title')
                    ->label(__('codoc-title'))
                    ->icon('fas-circle-info')
                    ->iconPosition(IconPosition::After)
                    ->tooltip(fn($record) => $record->type->getLabel())
                    ->sortable()
                    ->searchable(),

                Tables\Columns\IconColumn::make('service_url')
                    ->openUrl(),

                Tables\Columns\TextColumn::make('organisationProviding.title')
                    ->label(__('codoc-supplier'))
                    ->url(fn (Service $record): string
                        => OrganisationPageResource::getUrl('edit', ['record' => $record->organisation_provided_id ?? '#'])),

                Tables\Columns\TextColumn::make('service_status')
                    ->label(__('codoc-status'))
                    ->badge()
            ])

            ->filters([
                Tables\Filters\SelectFilter::make('codoc_organisation_uses_services.id')
                    ->label(__('id'))
                    ->options(Service::onlyMyRecords()->orderBy('title')->pluck('id', 'id')),

                Tables\Filters\SelectFilter::make('title')
                    ->label(__('codoc-service'))
                    ->options(Service::onlyMyRecords()->orderBy('title')->pluck('title', 'title')),

                Tables\Filters\SelectFilter::make('organisation_provided_id')
                    ->label(__('codoc-supplier'))
                    ->options(Organisation::onlyMyRecords()->orderBy('title')->pluck('title', 'id')),

                Tables\Filters\SelectFilter::make('organisation_type')
                    ->label(__('codoc-organisation-type'))
                    ->options(OrganisationType::class)
                    ->query(fn(Builder $query, $state) =>
                        $query->whereHas('organisationsProviding', fn($q1)
                                => $state['value'] === null ? $q1 : $q1->where('type', $state))),

                Tables\Filters\SelectFilter::make('type')
                    ->label(__('codoc-service-type'))
                    ->options(ServiceType::class),

                Tables\Filters\Filter::make('dpa_not_present')
                    ->label(__('codoc-dpa-not-present'))
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->hasNoDpa())
            ], layout: Tables\Enums\FiltersLayout::AboveContent)
            ->filtersFormColumns(5)

            ->headerActions([
                Tables\Actions\AttachAction::make()
                    ->color(Color::Yellow)
                    ->label(__('codoc-service-assign'))
                    ->recordSelect(
                        fn (Select $select) =>
                            $select
                                ->autofocus()
                                ->placeholder('Service auswählen')
                    )
                    ->recordSelectSearchColumns(['title', 'key'])
                    ->recordSelectOptionsQuery(fn(Builder $query)
                        => $query->onlyMyRecords()->active()->reorder('title', 'ASC'))
                    ->preloadRecordSelect()

                    ->mutateFormDataUsing(function (array $data): array
                    {
                        $serviceId = $data['recordId'];
                        $organisationId = $data['organisation_provided_id'];

                        $service = OrganisationProvidesServices::where('service_id', $serviceId)
                            ->where('organisation_id', $organisationId)->first();

                        if($service)
                        {
                            $data['service_title'] = $service->service_title;
                            $data['service_description'] = $service->service_description;
                            $data['service_url'] = $service->service_url;
                        }

                        return $data;
                    })

                    ->form(fn (Tables\Actions\AttachAction $action): array => [
                        $action->getRecordSelect(),

                        Forms\Components\Select::make('organisation_provided_id')
                            ->label(__('codoc-supplier'))
                            ->options(function($get)
                            {
                                $selectedServiceId = $get('recordId');

                                return Organisation::supplierOrOwnCompany()
                                    ->whereHas('servicesProviding', function($query) use ($selectedServiceId) {
                                        return $query->where('service_id', $selectedServiceId);
                                    })
                                    ->orderBy('title')
                                    ->get()
                                    ->pluck('title', 'id');
                            })
                            ->required()
                            ->searchable(),
                    ])
            ])

            ->actions([
                Tables\Actions\EditAction::make()
                    ->label('')
                    ->tooltip('Bearbeiten')
                    ->iconSize(IconSize::Medium)
                    ->modalWidth(MaxWidth::FiveExtraLarge)
                    ->mutateRecordDataUsing(function($record, $data)
                    {
                        if(empty($data['service_title']))
                        {
                            $data['service_title'] = $data['title'];
                        }

                        if(empty($data['service_description']))
                        {
                            $data['service_description'] = $data['description'];
                        }

                        if(empty($data['service_url']))
                        {
                            $data['service_url'] = $record->organisationProviding->homepage ?? '';
                        }

                        if(empty($data['service_additional']))
                        {
                            $data['service_additional'] = $data['additional_template'] ?? [];
                        }

                        return $data;
                    }),

                Tables\Actions\Action::make('edit_service')
                    ->label('')
                    ->icon('heroicon-o-arrow-right-start-on-rectangle')
                    ->color(Color::Orange)
                    ->iconSize(IconSize::Medium)
                    ->tooltip(__('codoc-edit-service'))
                    ->url(fn (Service $record): string => ServicePageResource::getUrl('edit', ['record' => $record->service_id])),

                Tables\Actions\DetachAction::make()
                    ->label('')
                    ->tooltip('Trennen')
                    ->iconSize(IconSize::Medium)
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DetachBulkAction::make()
                ]),
            ])

            ->defaultSort('sort_order', 'asc')
            ->reorderable('sort_order', true)
            ->reorderRecordsTriggerAction(
                fn (Tables\Actions\Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? __('codoc-sort-save') : __('codoc-sort'))
                    ->color($isReordering ? 'warning' : 'gray')
            );
    }

    #[On('deeplink-servicesusing')]
    public function executeDeeplink(string $action, string $id)
    {
        $this->replaceMountedTableAction($action, $id);
    }
}
