<?php

namespace Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconSize;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Netzperfekt\SaasCodoc\Filament\Components\Comments;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\ServicePageResource;
use Netzperfekt\SaasCodoc\Models\Service;

class ServiceRelationManager extends RelationManager
{
    protected static string $relationship = 'components';
    protected static ?string $inverseRelationship = 'components';
    protected static bool $isLazy = false;

    public static function getRecordLabel(): string
    {
        return __('codoc-service-component');
    }

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        $count = $ownerRecord->components->count();

        return __('codoc-services-components-count', ['count' => $count > 0 ? '[' . $count . ']' : '']);
    }

    public function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                TextInput::make('component_title')
                    ->required(),

                RichEditor::make('component_description')
                    ->defaultRichEditor(),

                Comments::make('component_comments')
                    ->user(Auth::user())
                    ->editableOnlyOwnValues()
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('title')
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('codoc-title'))
                    ->weight(FontWeight::SemiBold)
                    ->sortable()
                    ->searchable(),

                /*
                Tables\Columns\TextColumn::make('key')
                    ->label(__('codoc-key'))
                    ->sortable()
                    ->searchable(),
                */

                Tables\Columns\TextColumn::make('component_title')
                    ->label(__('codoc-description'))
                    ->sortable()
                    ->searchable(),
            ])

            ->filters([
                //
            ])

            ->headerActions([
                Tables\Actions\AttachAction::make()
                    ->color(Color::Yellow)
                    ->label(__('codoc-component-assign'))
                    ->recordSelect(
                        fn (Select $select) => $select
                            ->placeholder('Service-Komponente auswählen')
                            ->autofocus()
                            // ->multiple() // TODO hier klappt was bei apple/safari nicht - die buttons sind nicht erreichbar (css probleme?)
                    )
                    ->recordSelectOptionsQuery(fn(Builder $query) => $query->onlyMyRecords())
                    ->preloadRecordSelect()
            ])

            ->actions([
                Tables\Actions\EditAction::make()
                    ->label('')
                    ->tooltip('Bearbeiten')
                    ->iconSize(IconSize::Medium)
                    ->mutateRecordDataUsing(function($data)
                    {
                        if(empty($data['component_title']))
                        {
                            $data['component_title'] = $data['title'] ;
                        }

                        if(empty($data['component_description']))
                        {
                            $data['component_description'] = $data['description'] ;
                        }

                        return $data;
                    }),

                Tables\Actions\Action::make('edit_organisation')
                    ->label('')
                    ->icon('heroicon-o-arrow-right-start-on-rectangle')
                    ->color(Color::Orange)
                    ->iconSize(IconSize::Medium)
                    ->tooltip(__('codoc-edit-service'))
                    ->url(fn (Service $record): string => ServicePageResource::getUrl('edit', ['record' => $record->component_id])),

                Tables\Actions\DetachAction::make()
                    ->label('')
                    ->tooltip('Trennen')
                    ->iconSize(IconSize::Medium)
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DetachBulkAction::make()
                ]),
            ])

            ->defaultSort('sort_order', 'asc')
            ->reorderable('sort_order', true)
            ->reorderRecordsTriggerAction(
                fn (Tables\Actions\Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? __('codoc-sort-save') : __('codoc-sort'))
                    ->color($isReordering ? 'warning' : 'gray')
            );
    }
}
