<?php

namespace Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource;

use Carbon\Carbon;
use Filament\Facades\Filament;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Support\Colors\Color;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasCodoc\Enums\ServiceDocType;

class ServiceDocumentation
{
    public static function getForm(): array
    {
        return [
            Repeater::make('docs')
                ->label('')
                ->itemLabel(function ($state) {
                    $type = ServiceDocType::from($state['type'])->getLabel();

                    $s = '<div class="flex">';
                    $s .= '<div class="p-1 text-info-700 dark:text-info-400 font-semibold">' . Carbon::create($state['created_at'])->format('d.m.Y') . '</div>';
                    $s .= '<div class="p-1 ml-2">' . $state['title'] . '</div>';
                    $s .= '<div class="p-1 text-info-700 dark:text-info-400 ml-2">' . $state['user'] . '</div>';
                    $s .= '<div class="px-2 py-1 bg-info-200 text-gray-800 font-normal ml-4">' . $type . '</div>';
                    $s .= '</div>';

                    // TODO flex / ml-auto klappt hier nicht richtig - hätte gerne den 'type' am rechten rand...

                    return new HtmlString($s);
                })
                ->extraAttributes(['class' => 'repeater-docs'])
                ->addActionLabel(__('codoc-add-doc'))
                ->relationship('docs')
                ->defaultItems(0)
                ->deleteAction(
                    fn(Action $action) => $action->requiresConfirmation(),
                )
                ->collapsible()->collapsed()
                ->schema([
                    Grid::make()
                        ->columns(15)
                        ->schema([
                            TextInput::make('title')
                                ->label(__('codoc-title'))
                                ->columnSpan(8)
                                ->maxLength(256)
                                ->extraAttributes(['class' => 'font-semibold'])
                                ->autofocus()
                                ->live(),

                            Select::make('type')
                                ->label(__('codoc-type'))
                                ->columnStart(10)
                                ->columnSpan(2)
                                ->options(ServiceDocType::class)
                                ->selectablePlaceholder(false)
                                ->default(ServiceDocType::Default->value)
                                ->live(),

                            TextInput::make('ticketid')
                                ->label(__('codoc-ticketid'))
                                ->columnSpan(2)
                                ->maxLength(256),

                            DatePicker::make('created_at')
                                ->label(__('codoc-date'))
                                ->columnSpan(2)
                                ->native(false)
                                ->displayFormat('d.m.Y')
                                ->live()
                                ->default(Carbon::now()),

                            TextInput::make('user')
                                ->formatStateUsing(fn($record) => $record?->user?->name ?? Auth::user()->name)
                                ->visible(false),

                            Grid::make()
                                ->visible(fn($get) => $get('type') == ServiceDocType::Default->value)
                                ->schema(self::getFormForDocTypeDefault()),

                            Grid::make()
                                ->visible(fn($get) => $get('type') == ServiceDocType::Modification->value)
                                ->schema(self::getFormForDocTypeModification()),

                            Grid::make()
                                ->visible(fn($get) => $get('type') == ServiceDocType::Decision->value)
                                ->schema(self::getFormForDocTypeDecision()),

                            Grid::make()
                                ->visible(fn($get) => $get('type') == ServiceDocType::Attachments->value)
                                ->columns(15)
                                ->schema(self::getFormForDocTypeAttachments()),

                            Grid::make()
                                ->visible(fn($get) => $get('type') == ServiceDocType::Maintenance->value)
                                ->schema(self::getFormForDocTypeMaintenante()),
                        ])
                ])
        ];
    }

    private static function getFormForDocTypeDefault(): array
    {
        return [
            RichEditor::make('documentation')
                ->label(__('codoc-documentation'))
                ->columnSpanFull()
                ->defaultRichEditor(),
        ];
    }

    private static function getFormForDocTypeModification(): array
    {
        return [
            RichEditor::make('extra.why')
                ->label(__('codoc-why'))
                ->columnSpanFull()
                ->defaultRichEditor(),

            RichEditor::make('documentation')
                ->label(__('codoc-what'))
                ->columnSpanFull()
                ->defaultRichEditor(),

            RichEditor::make('extra.notes')
                ->label(__('codoc-notes'))
                ->columnSpanFull()
                ->defaultRichEditor()
        ];
    }

    private static function getFormForDocTypeDecision(): array
    {
        return [
            RichEditor::make('extra.why')
                ->label(__('codoc-decision-why'))
                ->columnSpanFull()
                ->defaultRichEditor(),

            RichEditor::make('documentation')
                ->label(__('codoc-decision-what'))
                ->columnSpanFull()
                ->defaultRichEditor(),

            RichEditor::make('extra.justification')
                ->label(__('codoc-decision-justification'))
                ->columnSpanFull()
                ->defaultRichEditor()
        ];
    }

    private static function getFormForDocTypeAttachments(): array
    {
        return [
            RichEditor::make('documentation')
                ->label(__('codoc-attachments-description'))
                ->columnSpanFull()
                ->columnSpan(8)
                ->defaultRichEditor(),

            FileUpload::make('extra.attachments')
                ->label(__('codoc-attachments'))
                ->hint(new HtmlString(__('codoc-attachments-hint')))
                ->hintColor(Color::Orange)
                ->multiple()
                ->directory(fn($record) => 'files/t' . Filament::getTenant()->id . '_s' . $record->id)
                ->disk(config('saas-codoc.attachments_disk'))
                ->maxSize(config('saas-codoc.attachments_max_size'))
                ->visibility('private')
                ->downloadable(true)
                ->openable(true)
                ->moveFiles()
                ->preserveFilenames()
                ->columnStart(10)
                ->columnSpan(6),
        ];
    }

    private static function getFormForDocTypeMaintenante(): array
    {
        return [
            Grid::make()
                ->columns(4)
                ->schema([
                    Textarea::make('documentation')
                        ->label(__('codoc-maintenance-components'))
                        ->columnSpan(2)
                        ->rows(2)->autosize(),

                    TextInput::make('extra.version_from')
                        ->label(__('codoc-maintenance-version-from')),

                    TextInput::make('extra.version_to')
                        ->label(__('codoc-maintenance-version-to')),
                ]),

            RichEditor::make('extra.problems')
                ->label(__('codoc-maintenance-problems'))
                ->columnSpanFull()
                ->defaultRichEditor(),

            RichEditor::make('extra.notes')
                ->label(__('codoc-maintenance-notes'))
                ->columnSpanFull()
                ->defaultRichEditor()
        ];
    }
}