<?php

namespace Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource;

use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\KeyValue;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Guava\FilamentKnowledgeBase\Contracts\HasKnowledgeBase;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Netzperfekt\SaasCodoc\Enums\ServiceStatus;
use Netzperfekt\SaasCodoc\Enums\ServiceType;
use Netzperfekt\SaasCodoc\Filament\Components\Comments;
use Netzperfekt\SaasCodoc\Filament\Components\CommentsColumn;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\Pages\CreateServicePage;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\Pages\EditServicePage;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\Pages\ListServicePages;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\Pages\ViewServicePage;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\RelationManagers\OrganisationProvidesRelationManager;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\RelationManagers\OrganisationUsesRelationManager;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\RelationManagers\ServiceRelationManager;
use Netzperfekt\SaasCodoc\Models\Service;

class ServicePageResource extends Resource implements HasKnowledgeBase
{
    protected static ?string $model = Service::class;

    protected static ?string $navigationIcon = 'heroicon-o-arrow-right-start-on-rectangle';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = -1;

    public static function getNavigationGroup(): ?string
    {
        return __('codoc-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return __('codoc-service');
    }

    public static function getPluralModelLabel(): string
    {
        return __('codoc-services');
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['title', 'key'];
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Typ' => $record->type->getLabel()
        ];
    }

    public static function getFormFields(bool $withAssignToOwnOrganisationButton = false): array
    {
        return [
            Split::make([
                Section::make(fn($get) => $get('type') == ServiceType::Project->value ? __('codoc-project') : __('codoc-service'))
                    ->schema([
                        Grid::make()
                            ->columns(4)
                            ->schema([
                                TextInput::make('key')
                                    ->label(__('codoc-key'))
                                    ->required()
                                    ->maxLength(25)
                                    ->autofocus(),

                                TextInput::make('title')
                                    ->label(__('codoc-title'))
                                    ->columnSpan(2)
                                    ->required()
                                    ->maxLength(255)
                            ]),

                        Textarea::make('description')
                            ->label(__('codoc-description'))
                            ->rows(2)->autosize(),

                        Grid::make()
                            ->columns(2)
                            ->visible(fn($get) => $get('type') == ServiceType::Project->value)
                            ->schema([
                                TextInput::make('url_project')
                                    ->label(__('codoc-project-url'))
                                    ->hint(__('codoc-project-url-hint'))
                                    ->url()
                                    ->maxLength(255)
                                    ->live()
                                    ->suffixAction(Action::make('open-project-url')
                                        ->icon('fas-link')
                                        ->url(fn($get) => $get('url_project'))
                                        ->disabled(fn($get) => $get('url_project') == '')
                                        ->openUrlInNewTab()),

                                TextInput::make('url_repository')
                                    ->label(__('codoc-repository-url'))
                                    ->url()
                                    ->live()
                                    ->maxLength(255)
                                    ->suffixAction(Action::make('open-repository-url')
                                        ->icon('fas-link')
                                        ->url(fn($get) => $get('url_repository'))
                                        ->disabled(fn($get) => $get('url_repository') == '')
                                        ->openUrlInNewTab()),

                                TextInput::make('url_documentation')
                                    ->label(__('codoc-documentation-url'))
                                    ->hint(__('codoc-documentation-url-hint'))
                                    ->url()
                                    ->live()
                                    ->maxLength(255)
                                    ->suffixAction(Action::make('open-documentation-url')
                                        ->icon('fas-link')
                                        ->url(fn($get) => $get('url_documentation'))
                                        ->disabled(fn($get) => $get('url_documentation') == '')
                                        ->openUrlInNewTab()),

                                TextInput::make('url_projectmanagement')
                                    ->label(__('codoc-management-url'))
                                    ->url()
                                    ->live()
                                    ->maxLength(255)
                                    ->suffixAction(Action::make('open-management-url')
                                        ->icon('fas-link')
                                        ->url(fn($get) => $get('url_projectmanagement'))
                                        ->disabled(fn($get) => $get('url_projectmanagement') == '')
                                        ->openUrlInNewTab()),

                                Textarea::make('description_deployment')
                                    ->label(__('codoc-description-deployment'))
                                    ->rows(1)->autosize()
                                    ->columnSpanFull(),
                            ])
                    ])
                    ->collapsible()
                    ->persistCollapsed()
                    ->grow(),

                Section::make([
                    Select::make('type')
                        ->label(__('codoc-type'))
                        ->options(ServiceType::class)
                        ->live()
                        ->required(),

                    Select::make('status')
                        ->label(__('codoc-status'))
                        ->options(fn($get) => ServiceStatus::getOptionsForType(ServiceType::from((int)$get('type'))))
                        ->default(ServiceStatus::Active)
                        ->required(),

                    TagsInput::make('tags')
                        ->label(__('codoc-tags'))
                        ->splitKeys([','])
                        ->maxWidth(MaxWidth::ExtraSmall),

                ])->grow(false)
            ])->from('xl'),

            // 24.06.24 CF/NE doch eher keine gesonderte projekt-doku -> suggeriert zu stark eine "projektverwaltung"
            /*
            // TODO "dokumentationseintrag hinzufügen" -> ggf. nach rechts + icon "plus symbol"
            // TODO doku eintrag hinzufügen, leer lassen, form wird nicht richtig validiert (keine anzeige der validierungsfehler -> mysql error)
            Section::make(__('codoc-docs'))
                ->visible(fn($get) => $get('type') == ServiceType::Project->value)
                ->schema(ServiceDocumentation::getForm())
                ->compact()
                ->collapsible()
                ->persistCollapsed(),
            */

            Section::make(__('codoc-comments'))
                // 24.06.24 CF/NE doch eher nicht (s.o.)
                // ->visible(fn($get) => $get('type') != ServiceType::Project->value)
                ->schema([
                    Comments::make('comments')
                        ->label('')
                        ->user(Auth::user()),
                ])
                ->compact()
                ->collapsible()
                ->persistCollapsed(),

            KeyValue::make('additional_template')
                ->label(__('codoc-additional-template'))
                ->hintIcon(
                    'heroicon-m-question-mark-circle',
                    tooltip: __('codoc-additional-template-help')
                )
                ->valueLabel(__('codoc-additional-template-value'))
                ->addActionLabel(__('codoc-additional-template-add'))
                ->default(ServiceType::getDefaultAdditionalFieldTemplate()),

            Toggle::make('assignToOwnOrganisation')
                ->label(__('codoc-assign-to-own-organisation'))
                ->visible($withAssignToOwnOrganisationButton)
        ];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema(self::getFormFields());
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultPaginationPageOption(25)
            ->groups([
                Tables\Grouping\Group::make('type')
                    ->label(__('codoc-type'))
                    ->collapsible(),

                Tables\Grouping\Group::make('status')
                    ->label(__('codoc-status'))
                    ->collapsible()
            ])
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('codoc-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('components_count')
                    ->counts('components')
                    ->label(__('codoc-services-components'))
                    ->badge()
                    ->color(Color::Orange)
                    ->formatStateUsing(fn($state) => $state ? $state : '')
                    ->tooltip(function ($record) {
                        return $record->components->pluck('title')->implode(' | ');
                    }),

                Tables\Columns\TextColumn::make('organisations_using_count')
                    ->label(__('codoc-organisations-using'))
                    ->countWithMissingAlert(
                        function ($record) {
                            return ($componentCount = $record->isComponentFromServiceCount()) ? $componentCount : $record->organisationsUsing->count();
                        },
                        fn($record) => __('codoc-organisations-using')
                    )
                    ->modalOrganisationsUsing(),

                Tables\Columns\TextColumn::make('organisations_providing_count')
                    ->label(__('codoc-organisations-providing'))
                    ->countWithMissingAlert(
                        fn($record) => $record->organisationsProviding->count(),
                        fn($record) => __('codoc-organisations-providing')
                    )
                    ->modalOrganisationsProviding(),

                CommentsColumn::make('comments')
                    ->tooltip(__('codoc-comments'))
                    ->showCount(),

                Tables\Columns\TextColumn::make('type')
                    ->label(__('codoc-type'))
                    ->color(Color::Green)
                    ->badge(),

                Tables\Columns\TextColumn::make('status')
                    ->label(__('codoc-status'))
                    ->badge()
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('type')
                    ->label(__('codoc-type'))
                    ->options(ServiceType::class),

                Tables\Filters\SelectFilter::make('status')
                    ->label(__('codoc-status'))
                    ->options(ServiceStatus::class),

                Tables\Filters\SelectFilter::make('without_associations')
                    ->indicator('without_associations')
                    ->indicateUsing(fn(array $state) => $state['without_associations'] ? __('codoc-without-associations') : null)
                    ->form([
                        Toggle::make('without_associations')->label(__('codoc-without-associations'))

                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['without_associations'],
                                fn(Builder $query, $date): Builder => $query->doesntHave('organisationsProviding')
                                    ->orDoesntHave('organisationsUsing')
                                    ->withoutComponents()
                                    ->withoutTrashed()
                            );
                    })
            ])
            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),

                Tables\Actions\DeleteAction::make()
                    ->iconButton()
                    ->tooltip('Löschen'),

                Tables\Actions\ForceDeleteAction::make()
                    ->iconButton()
                    ->tooltip('Endgültig löschen')
                    ->before(function (Tables\Actions\ForceDeleteAction $action, Service $record) {
                        if ($record->organisationsProviding->count() > 0 || $record->organisationsUsing->count() > 0) {
                            Notification::make()
                                ->danger()
                                ->title(__('codoc-msg-failed'))
                                ->body(__('codoc-msg-service-assigned'))
                                ->persistent()
                                ->send();

                            $action->cancel();
                        }
                    }),

                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ])
            ->defaultSort('title');
    }

    public static function getRelations(): array
    {
        return [
            OrganisationProvidesRelationManager::class,
            OrganisationUsesRelationManager::class,
            ServiceRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListServicePages::route('/'),
            'create' => CreateServicePage::route('/create'),
            'edit' => EditServicePage::route('/{record}/edit')
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }


    public static function getDocumentation(): array
    {
        return [
            'services'
        ];
    }
}
