<?php

namespace Netzperfekt\SaasCodoc\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasBase\Traits\HasDefaultScopes;
use Netzperfekt\SaasBase\Traits\HasGlobalModelObserver;
use Netzperfekt\SaasCodoc\Enums\OrganisationStatus;
use Netzperfekt\SaasCodoc\Enums\OrganisationType;
use Netzperfekt\SaasCodoc\Enums\ServiceStatus;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

class Organisation extends Model
{
    use SoftDeletes;
    use HasGlobalModelObserver;
    use HasDefaultScopes;
    use LogsActivity;

    protected $table = 'codoc_organisations';

    protected $fillable = [
        'id', 'deleted_at', 'created_at', 'updated_at',
        'team_id',
        'type', 'subtype',
        'authorization_tags',
        'status',
        'title',
        'company1',
        'company2',
        'street',
        'zip',
        'city',
        'country',
        'phone',
        'email',
        'homepage',
        'contact',
        'additional',
        'comments',
        'documents',
        'assigned_organisations',
        'is_favorite'
    ];

    protected $casts = [
        'authorization_tags' => 'array',
        'additional'         => 'array',
        'comments'           => 'array',
        'documents'          => 'array',
        'assigned_organisations' => 'array',
        'type'               => OrganisationType::class,
        'subtype'            => OrganisationType::class,
        'status'             => OrganisationStatus::class
    ];

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function servicesProviding(): BelongsToMany
    {
        return $this->belongsToMany(
            Service::class,
            'codoc_organisation_provides_services',
            'organisation_id',
            'service_id'
        )
            ->withPivot(['id',
                         'service_title', 'service_description', 'service_url', 'service_status',
                         'service_needs_dpa', 'service_documents', 'service_comments', 'service_additional',
                         'is_favorite', 'sort_order'])
            ->withCasts([
                'service_comments'      => 'array',
                'service_documents'     => 'array',
                'service_additional'    => 'array',
                'service_status'        => ServiceStatus::class
            ])
            ->using(OrganisationProvidesServices::class)
            ->orderBy('sort_order');
    }

    public function servicesUsing(): BelongsToMany
    {
        return $this->belongsToMany(
            Service::class,
            'codoc_organisation_uses_services',
            'organisation_id',
            'service_id'
        )
            ->withPivot(['id', 'organisation_provided_id',
                         'service_title', 'service_description', 'service_url', 'service_status',
                         'service_needs_dpa', 'service_documents', 'service_comments', 'service_additional',
                         'is_favorite', 'sort_order'])
            ->withCasts([
                'service_comments'      => 'array',
                'service_documents'     => 'array',
                'service_additional'    => 'array',
                'service_status'        => ServiceStatus::class
            ])
            ->using(OrganisationUsesServices::class)
            ->orderBy('sort_order');
    }

    public function assets(): HasMany
    {
        return $this->hasMany(Asset::class);
    }

    public function scopeOwnCompany(Builder $query): void
    {
        $query->onlyMyRecords()->whereIn('type', [ OrganisationType::OwnOrganisation ]);
    }

    public function scopeWithoutOwnCompany(Builder $query): void
    {
        $query->onlyMyRecords()->whereNotIn('type', [ OrganisationType::OwnOrganisation ]);
    }

    public function scopeWithoutAssociations(Builder $query): void
    {
        $query->doesntHave('servicesProviding')
              ->doesntHave('servicesUsing');
    }

    public function scopeSupplierOrOwnCompany(Builder $query): void
    {
        $query->whereIn('type', OrganisationType::getSupplier());
    }

    public function isMyself(): bool
    {
        return $this->isOwnCompany();
    }

    public function isOwnCompany(): bool
    {
        return $this->type == OrganisationType::OwnOrganisation;
    }

    public function isOwnProduct(): bool
    {
        return $this->type == OrganisationType::OwnProduct;
    }

    public function toggleFavorite(): void
    {
        $this->is_favorite = ! $this->is_favorite;
    }

    public static function getOwnCompany(): ?Organisation
    {
        return self::ownCompany()->first();
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['status', 'comments', 'documents'])
            ->dontLogIfAttributesChangedOnly(['updated_at'])
            ->dontSubmitEmptyLogs()
            ->logOnlyDirty(true);
    }
}
