<?php

namespace Netzperfekt\SaasCodoc\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasBase\Traits\HasDefaultScopes;
use Netzperfekt\SaasBase\Traits\HasGlobalModelObserver;
use Netzperfekt\SaasCodoc\Enums\ServiceStatus;
use Netzperfekt\SaasCodoc\Enums\ServiceType;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

class Service extends Model
{
    use SoftDeletes;
    use HasGlobalModelObserver;
    use HasDefaultScopes;
    use LogsActivity;

    protected $table = 'codoc_services';

    protected $fillable = [
        'id', 'deleted_at', 'created_at', 'updated_at',
        'team_id',
        'type',
        'key',
        'title',
        'description',
        'tags',
        'status',
        'url_project',
        'url_repository',
        'url_documentation',
        'url_projectmanagement',
        'description_deployment',
        'comments',
        'additional_template'
    ];

    protected $casts = [
        'tags'                => 'array',
        'comments'            => 'array',
        'additional_template' => 'array',
        'type'                => ServiceType::class,
        'status'              => ServiceStatus::class
    ];

    public function scopeProjects(Builder $query): void
    {
        $query
            ->whereIn('type', [ ServiceType::Project ])
            ->orderBy('title');
    }

    public function scopeActive(Builder $query): void
    {
        $query->whereIn('status', [ ServiceStatus::Active, ServiceStatus::Current, ServiceStatus::Productive ]);
    }

    public function scopeWithoutComponents(Builder $query): void
    {
        $query->doesntHave('parentServices');
    }

    public function scopeWithoutAssociations(Builder $query): void
    {
        $query->doesntHave('organisationsProviding')
              ->orDoesntHave('organisationsUsing');
    }

    public function scopeHasNoDpa(Builder $query): void
    {
        $query->onlyMyRecords()->where('service_needs_dpa', false);
    }


    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function components(): BelongsToMany
    {
        return $this->belongsToMany(
            Service::class,
            'codoc_services_components',
            'service_id',
            'component_id'
        )
            ->withPivot(['id', 'component_title', 'component_description', 'component_comments', 'sort_order'])
            ->withCasts(['component_comments' => 'array'])
            ->using(ServicesComponents::class);
    }

    public function parentServices(): BelongsToMany
    {
        return $this->belongsToMany(
            Service::class,
            'codoc_services_components',
            'component_id',
            'service_id'
        )
            ->withPivot(['id', 'component_title', 'component_description', 'component_comments', 'sort_order'])
            ->withCasts(['component_comments' => 'array'])
            ->using(ServicesComponents::class);
    }

    public function isComponentFromServiceCount(): int
    {
        return ServicesComponents::where('component_id', $this->id)->count();
    }

    public function service(): BelongsTo
    {
        return $this->belongsTo(Service::class, 'service_id', 'id');
    }

    public function organisationsProviding(): BelongsToMany
    {
        return $this->belongsToMany(
            Organisation::class,
            'codoc_organisation_provides_services',
            'service_id',
            'organisation_id'
        )
            ->withPivot(['id',
                         'service_title', 'service_description', 'service_url', 'service_status',
                         'service_needs_dpa', 'service_documents', 'service_comments', 'service_additional',
                         'sort_order'])
            ->withCasts([
                'service_comments'      => 'array',
                'service_documents'     => 'array',
                'service_additional'    => 'array',
                'service_status'        => ServiceStatus::class
            ])
            ->using(OrganisationProvidesServices::class)
            ->orderBy('title');
    }

    public function organisationProviding(): BelongsTo
    {
        return $this->belongsTo(Organisation::class, 'organisation_provided_id', 'id');
    }

    public function organisationsUsing(): BelongsToMany
    {
        return $this->belongsToMany(
            Organisation::class,
            'codoc_organisation_uses_services',
            'service_id',
            'organisation_id'
        )
            ->withPivot(['id', 'organisation_provided_id',
                         'service_title', 'service_description', 'service_url', 'service_status',
                         'service_needs_dpa', 'service_documents', 'service_comments', 'service_additional',
                         'sort_order'
            ])
            ->withCasts([
                'service_comments'      => 'array',
                'service_documents'     => 'array',
                'service_additional'    => 'array',
                'service_status'        => ServiceStatus::class])
            ->using(OrganisationUsesServices::class)
            ->orderBy('title');
    }

    public function docs(): HasMany
    {
        return $this->hasMany(ServiceDoc::class)->orderBy('created_at', 'DESC');
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['status', 'comments'])
            ->dontLogIfAttributesChangedOnly(['updated_at'])
            ->dontSubmitEmptyLogs()
            ->logOnlyDirty(true);
    }
}
