<?php

namespace Netzperfekt\SaasCodoc;

use Filament\Infolists\Components\RepeatableEntry;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Support\Assets\AlpineComponent;
use Filament\Support\Assets\Css;
use Filament\Support\Assets\Js;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Facades\FilamentAsset;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\Column;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasCodoc\Commands\ExportCodocCommand;
use Netzperfekt\SaasCodoc\Commands\ImportCodocCommand;
use Netzperfekt\SaasCodoc\Filament\Resources\OrganisationPageResource\OrganisationPageResource;
use Netzperfekt\SaasCodoc\Filament\Resources\ServicePageResource\ServicePageResource;
use Spatie\LaravelPackageTools\Commands\InstallCommand;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;

class SaasCodocServiceProvider extends PackageServiceProvider
{
    public function packageRegistered()
    {
        $this->app->register(EventServiceProvider::class);
    }

    public function packageBooted()
    {
        $this->addAssets();
        $this->addDocs();
        $this->addMacros();
    }

    public function configurePackage(Package $package): void
    {
        $package
            ->name(SaasCodoc::PACKAGE_NAME)
            ->hasRoute('web')
            ->hasViews()
            ->hasConfigFile()
            ->hasTranslations()

            ->hasMigrations([
                'create_organisations_table',
                'create_services_table',
                'create_service_components_table',
                'create_service_doc_table',
                'create_organisation_provides_services_table',
                'create_organisation_uses_services_table',
                'create_assets_table',
                'create_export_links_table'
            ])
            
            ->hasCommands(
                ExportCodocCommand::class,
                ImportCodocCommand::class
            )

            ->hasInstallCommand(function(InstallCommand $command)
            {
                $command
                    ->publishAssets()
                    ->publishMigrations()
                    ->publishConfigFile()
                    ->askToRunMigrations();
            });
    }

    private function addAssets()
    {
        FilamentAsset::register([
            Js::make('saas-codoc-js', __DIR__ . '/../resources/dist/saas-codoc.js'),
            Css::make('saas-codoc-css', __DIR__ . '/../resources/dist/saas-codoc.css'),
            Css::make('saas-codoc-css-frontend', __DIR__ . '/../resources/dist/saas-codoc-frontend.css'),
            AlpineComponent::make('comments-field', __DIR__ . '/../resources/dist/comments-field.js'),
            AlpineComponent::make('dependency-field', __DIR__ . '/../resources/dist/dependency-field.js'),
        ], 'netzperfekt/saas-codoc');
    }

    private function addDocs()
    {
        $this->publishes([
            $this->package->basePath('/../docs') => base_path("/docs"),
        ], "{$this->package->shortName()}-docs");
    }

    private function addMacros()
    {
        Column::macro('showCount', function()
        {
            return $this
                ->label(__('codoc-comments'))
                ->action(
                    Action::make('show-comments')
                        ->modalContent(fn($record) => view(
                            'saas-codoc::filament.comments-modal',
                            ['comments' => $record->comments]) // TODO das ARRAY könnte noch absteigend nach 'date' sortiert werden
                        )
                        ->slideOver()
                        ->modalSubmitAction(false)
                        ->modalCancelActionLabel(__('OK'))
                );
        });

        Column::macro('tags', function()
        {
            return $this
                ->label(__('codoc-tags'))
                ->width('250px')
                ->searchable()
                ->badge()
                ->color(Color::Orange);
        });

        Column::macro('modalOrganisationsUsing', function() {
            return $this
                ->action(
                    Action::make('show_organisations_using')
                        ->modalCancelAction(false)
                        ->modalSubmitActionLabel(__('codoc-ok'))
                        ->infolist([
                            SaasCodocServiceProvider::getModalServicesHeader(),

                            RepeatableEntry::make('organisationsUsing')
                                ->label(__('codoc-organisations-using-label'))
                                ->columns(4)
                                ->visible(fn($record) => $record->isComponentFromServiceCount() == 0)
                                ->schema([
                                    TextEntry::make('title')
                                        ->label('')
                                        ->color(Color::Blue)
                                        ->weight(FontWeight::SemiBold)
                                        ->columnSpan(2)
                                        ->url(fn($record): string => OrganisationPageResource::getUrl('edit', ['record' => $record->id]))
                                        ->openUrlInNewTab(),

                                    TextEntry::make('type')
                                        ->label('')
                                        ->badge()
                                ]),

                            RepeatableEntry::make('parentServices')
                                ->label(new HtmlString(__('codoc-services-using-components-label')))
                                ->columns(4)
                                ->visible(fn($record) => $record->isComponentFromServiceCount() > 0)
                                ->schema([
                                    TextEntry::make('title')
                                        ->label('')
                                        ->color(Color::Blue)
                                        ->weight(FontWeight::SemiBold)
                                        ->columnSpan(2)
                                        ->url(fn($record): string => ServicePageResource::getUrl('edit', ['record' => $record->id]))
                                        ->openUrlInNewTab(),

                                    TextEntry::make('type')
                                        ->label('')
                                        ->badge()
                                ])
                        ])->label('')
                );
        });

        Column::macro('modalOrganisationsProviding', function() {
            return $this
                ->action(
                    Action::make('show_organisations_prodiving')
                        ->modalCancelAction(false)
                        ->modalSubmitActionLabel(__('codoc-ok'))
                        ->infolist([
                            SaasCodocServiceProvider::getModalServicesHeader(),

                            RepeatableEntry::make('organisationsProviding')
                                ->label(__('codoc-organisations-providing-label'))
                                ->columns(3)
                                ->schema([
                                    TextEntry::make('title')
                                        ->label('')
                                        ->color(Color::Blue)
                                        ->weight(FontWeight::SemiBold)
                                        ->columnSpan(2)
                                        ->url(fn($record): string => OrganisationPageResource::getUrl('edit', ['record' => $record->id]))
                                        ->openUrlInNewTab(),

                                    TextEntry::make('type')
                                        ->label('')
                                        ->badge()
                                ])
                        ])->label('')
                );
        });

        Column::macro('modalServicesProviding', function() {
            return $this
                ->action(
                    Action::make('show_services_prodiving')
                        ->modalCancelAction(false)
                        ->modalSubmitActionLabel(__('codoc-ok'))
                        ->infolist([
                            SaasCodocServiceProvider::getModalServicesHeader(),

                            RepeatableEntry::make('servicesProviding')
                                ->label(__('codoc-services-providing'))
                                ->columns(3)
                                ->schema([
                                    TextEntry::make('title')
                                        ->label('')
                                        ->color(Color::Blue)
                                        ->weight(FontWeight::SemiBold)
                                        ->columnSpan(2)
                                        ->url(fn($record): string => ServicePageResource::getUrl('edit', ['record' => $record->id]))
                                        ->openUrlInNewTab(),

                                    TextEntry::make('type')
                                        ->label('')
                                        ->badge()
                                ])
                        ])->label('')
                );
        });

        Column::macro('modalServicesUsing', function() {
            return $this
                ->action(
                    Action::make('show_services_using')
                        ->modalCancelAction(false)
                        ->modalSubmitActionLabel(__('codoc-ok'))
                        ->infolist([
                            SaasCodocServiceProvider::getModalServicesHeader(),

                            RepeatableEntry::make('servicesUsing')
                                ->label(__('codoc-services-using'))
                                ->columns(3)
                                ->schema([
                                    TextEntry::make('title')
                                        ->label('')
                                        ->color(Color::Blue)
                                        ->weight(FontWeight::SemiBold)
                                        ->columnSpan(2)
                                        ->url(fn($record): string => ServicePageResource::getUrl('edit', ['record' => $record->id]))
                                        ->openUrlInNewTab(),

                                    TextEntry::make('type')
                                        ->label('')
                                        ->badge()
                                ])
                        ])
                        ->label('')
                        ->stickyModalHeader()
                        ->stickyModalFooter()
                );
        });

        Column::macro('countWithMissingAlert', function(\Closure $callbackCount, \Closure $callbackTooltip)
        {
            $tooltipMissing = __('codoc-msg-no-association');

            return $this
                ->getStateUsing(fn($record) => ($count = $callbackCount($record)) > 0 ? $count : '---')

                ->icon(fn($state)
                    => $state > 0 ? '' : 'fas-exclamation-triangle')

                ->color(fn($state)
                    => $state > 0 ? Color::Yellow : Color::Red)

                ->tooltip(fn($state, $record)
                    => $state > 0 ? $callbackTooltip($record) : $tooltipMissing)

                ->badge();
        });

        Column::macro('openUrl', function(bool $openInNewTab = true)
        {
            return $this
                ->label('')
                ->icon(fn (?string $state): string => $state ? 'heroicon-o-link' : '')
                ->tooltip(fn ($state) => $state)
                ->url(fn (?string $state): string => $state ?? '', $openInNewTab)
                ->color('info');
        });
    }

    public static function getModalServicesHeader()
    {
        return Section::make()
            ->columns(3)
            ->compact()
            ->schema([
                TextEntry::make('title')
                    ->label('')
                    ->weight(FontWeight::SemiBold)
                    ->color(Color::Yellow),

                TextEntry::make('service_title')
                    ->label(''),

                TextEntry::make('type')
                    ->label(''),
            ]);
    }
}
