<?php

namespace Netzperfekt\SaasCodoc\Services;

use Filament\Notifications\Notification;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Netzperfekt\SaasCodoc\Models\Organisation;
use Spatie\LaravelPdf\Enums\Format;
use Spatie\LaravelPdf\Enums\Orientation;
use Spatie\LaravelPdf\Enums\Unit;
use Spatie\LaravelPdf\Facades\Pdf;
use Spatie\LaravelPdf\PdfBuilder;

class ExportService
{
    public function exportOrganisations(Collection $organisations, $params = [])
    {
        $params = [
            'organisations'     => $organisations,
            'myorganisation'    => Organisation::ownCompany()->get()->first(),
            'mydata'            => $params['mydata'] ?? false,
            'provided'          => $params['provided'] ?? false,
            'used'              => $params['used'] ?? false,
            'assets'            => $params['assets'] ?? false,
            'with_history'      => $params['history'] ?? false,
            'with_qrcode'       => $params['qrcode'] ?? false
        ];

        Notification::make()
            ->title(__('codoc-notification-downloaded'))
            ->success()
            ->send();

        Cookie::queue(Cookie::forever('_documentation_exported', 1));

        return $this->createExportPDF($params);
    }

    public function createExportPDF(array $params, string $userName = null): PdfBuilder
    {
        $userName = $userName ?? Auth::user()->name;

        return Pdf
            ::view('saas-codoc::pdf.organisations', [
                'codoc' => $params
            ])

            ->headerView('saas-codoc::pdf._header', [
                'header' => __('codoc-export-header', ['user' => $userName]),
            ])
            ->footerView('saas-codoc::pdf._footer', [
                'footer' => __('codoc-export-footer', ['user' => $userName])
            ])

            ->format(Format::A4)
            ->orientation(Orientation::Landscape)
            ->margins(2.5, 1.0, 1.5, 1.0, Unit::Centimeter);
    }

    public function exportDocumentation($params = [])
    {
        $ownOrganisation = Organisation::ownCompany()
            ->with(['servicesProviding', 'servicesUsing'])
            ->first();

        if( ! $ownOrganisation)
        {
            Notification::make()
                ->title(__('codoc-notification-no-organisation'))
                ->danger()
                ->send();

            return redirect()->back();
        }

        $codoc = [
            'ownOrganisation'   => $ownOrganisation,
            'servicesProviding' => $params['provided'] ? $ownOrganisation?->servicesProviding()->get() : [],
            'servicesUsing'     => $params['used'] ? $ownOrganisation?->servicesUsing()->get() : [],
            'assets'            => $params['assets'] ? $ownOrganisation?->assets()->get() : [],
            'with_history'      => $params['history'] ?? false,
            'with_qrcode'       => $params['qrcode'] ?? false
        ];

        Notification::make()
            ->title(__('codoc-notification-downloaded'))
            ->success()
            ->send();

        return Pdf
            ::view('saas-codoc::documentation.codoc', [
                'codoc' => $codoc
            ])

            ->headerView('saas-codoc::pdf._header', [
                'header' => 'CoDoc - Company Documentation'
            ])
            ->footerView('saas-codoc::pdf._footer', [
                'footer' => '*** Footer TODO ***'
            ])

            ->format(Format::A4)
            ->orientation(Orientation::Portrait)
            ->margins(2.5, 1.0, 1.5, 1.0, Unit::Centimeter);
    }
}
