<?php

namespace Database\Factories;

use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Carbon;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasDeadman\Enums\ConfirmationFrequency;
use Netzperfekt\SaasDeadman\Enums\ConfirmationFrequencyGrace;
use Netzperfekt\SaasDeadman\Enums\SwitchType;
use Netzperfekt\SaasDeadman\Models\DMSwitch;
use Netzperfekt\SaasDeadman\States\SwitchIdle;
use Netzperfekt\SaasDeadman\States\SwitchMustConfirm;

class DMSwitchFactory extends Factory
{
    protected $model = DMSwitch::class;

    public function definition(): array
    {
        $frequency = ConfirmationFrequency::Daily;

        return [
            'type' => SwitchType::Default,
            'title' => $this->faker->word(),
            'description' => $this->faker->text(),
            'active' => $this->faker->boolean(),
            'state' => SwitchIdle::class,
            'frequency' => $frequency,
            'frequency_grace' => ConfirmationFrequencyGrace::None,
            'confirmation_mode' => 0,
            'confirmation_owner_can_override' => false,
            'confirmations' => [],
            'confirmation_message_owner' => $this->faker->text(),
            'confirmation_message_additional' => $this->faker->text(),
            'pause_from' => null,
            'pause_until' => null,
            'last_confirmation' => null,
            'next_confirmation' => null,
            'next_confirmation_with_grace' => null,

            'team_id' => Team::factory(),
        ];
    }

    public function withTeam(Team $team): Factory
    {
        return $this->state(function (array $attributes) use ($team) {
            return [
                'team_id' => $team->id
            ];
        });
    }

    public function active(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'active' => true
            ];
        });
    }

    public function notActive(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'active' => false
            ];
        });
    }

    public function notPaused(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'pause_from' => null,
                'pause_until' => null,
            ];
        });
    }

    public function paused(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'pause_from' => Carbon::now()->subtract('1day'),
                'pause_until' => Carbon::now()->add('1day'),
            ];
        });
    }

    public function needsConfirmation(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'last_confirmation' => Carbon::now()->subtract('1day'),
                'next_confirmation' => Carbon::now()->subtract('1minute'),
                'next_confirmation_with_grace' => Carbon::now()->subtract('1minute')
            ];
        });
    }

    public function needsNoConfirmation(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'last_confirmation' => Carbon::now()->subtract('1day'),
                'next_confirmation' => Carbon::now()->add('10years'),
                'next_confirmation_with_grace' => Carbon::now()->add('10years')
            ];
        });
    }

    public function hourly(): Factory
    {
        return $this->state(function (array $attributes)
        {
            $frequency = \Netzperfekt\SaasDeadman\Enums\ConfirmationFrequency::Hourly;

            return [
                'frequency' => $frequency
            ];
        });
    }

    public function daily(): Factory
    {
        return $this->state(function (array $attributes)
        {
            $frequency = \Netzperfekt\SaasDeadman\Enums\ConfirmationFrequency::Daily;

            return [
                'frequency' => $frequency
            ];
        });
    }

    public function weekly(): Factory
    {
        return $this->state(function (array $attributes)
        {
            $frequency = \Netzperfekt\SaasDeadman\Enums\ConfirmationFrequency::Weekly;

            return [
                'frequency' => $frequency
            ];
        });
    }

    public function withoutGracePeriod(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'frequency_grace' => \Netzperfekt\SaasDeadman\Enums\ConfirmationFrequencyGrace::None,
            ];
        });
    }

    public function withGracePeriod4Hours(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'frequency_grace' => \Netzperfekt\SaasDeadman\Enums\ConfirmationFrequencyGrace::Hours4,
            ];
        });
    }

    public function stateIdle(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'state' => SwitchIdle::class,
            ];
        });
    }

    public function stateMustConfirm(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'state' => SwitchMustConfirm::class,
            ];
        });
    }

    public function overdue(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'next_confirmation' => Carbon::now()->subtract('1minute'),
                'next_confirmation_with_grace' => Carbon::now()->subtract('1minute')
            ];
        });
    }

    public function notOverdue(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'next_confirmation' => Carbon::now()->add('1day'),
                'next_confirmation_with_grace' => Carbon::now()->add('1day')
            ];
        });
    }

    public function confirmedByOwner(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'confirmations' => ['_0' => Carbon::now()]
            ];
        });
    }

    public function ownerCanOverrideConfirmers(): Factory
    {
        return $this->state(function (array $attributes) {
            return [
                'confirmation_owner_can_override' => true
            ];
        });
    }

    public function confirmationMode(int $mode): Factory
    {
        return $this->state(function (array $attributes) use ($mode) {
            return [
                'confirmation_mode' => $mode
            ];
        });
    }

    public function confirmedBy(array $contacts): Factory
    {
        return $this->state(function (array $attributes) use ($contacts)
        {
            $additionalContacts = [];
            foreach($contacts as $contact)
            {
                $additionalContacts['_' . $contact->id] = \Carbon\Carbon::now()->toString();
            }

            return [
                'confirmations' => [
                    ...$attributes['confirmations'],
                    ...$additionalContacts
                ]
            ];
        });
    }
}
