<?php

declare(strict_types=1);

namespace Netzperfekt\SaasDeadman\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasDeadman\Enums\ActivityLogType;
use Netzperfekt\SaasDeadman\Filament\Resources\ContentBlocks;
use Netzperfekt\SaasDeadman\Models\Action;

class DeleteUnusedAttachmentsCommand extends Command
{
    protected $signature = 'delete:unused-attachments';

    protected $description = 'Deletes unused attachments from storage provider';

    public function handle(): int
    {
        // TODO maybe refactor/move this to queues?

        $diskName = config('saas-deadman.attachments_disk_unencrypted');
        $actions = Action::all();

        $attachments = [];
        foreach($actions as $action)
        {
            if ($this->loginAsFirstUserOfTeam($action->team_id) === false)
            {
                continue;
            }

            foreach($action->contents as $block)
            {
                if($block['type'] == ContentBlocks::TYPE_ATTACHMENT_BLOCK)
                {
                    $attachments[] = $block['data']['attachment'] ?? '';
                }
            }
        }
        $attachments = array_unique($attachments);

        $countDeleted = 0;
        $allFilesOnStorage = Storage::disk($diskName)->allFiles();

        foreach($allFilesOnStorage as $fileName)
        {
            if( ! in_array($fileName, $attachments))
            {
                if(Storage::disk($diskName)->delete($fileName) === false)
                {
                    $this->warn('"' . $fileName . '" could not be deleted.');
                }
                else {
                    $countDeleted++;
                }
            }
        }

        Auth::logout();

        $this->info($countDeleted . ' attachments deleted.');

        if($countDeleted > 0)
        {
            activity()
                ->withProperties(['count' => $countDeleted])
                ->log(ActivityLogType::CommandLineInterface->value . '_' . $this->signature);
        }

        return self::SUCCESS;
    }

    private function loginAsFirstUserOfTeam(int $teamId): bool
    {
        $firstUserOfThisTeam = Team::where('id', $teamId)->first()->members()->first();
        if( ! $firstUserOfThisTeam)
        {
            return false;
        }

        $firstUserOfThisTeam->current_team_id = $teamId; // can be overwritten - is newly set at next login of this user
        $firstUserOfThisTeam->save();

        activity()->withoutLogs(fn () =>
            Auth::login($firstUserOfThisTeam)
        );

        return true;
    }
}
