<?php

namespace Netzperfekt\SaasDeadman\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Netzperfekt\SaasDeadman\Enums\ActivityLogType;
use Netzperfekt\SaasDeadman\Facades\SwitchService;
use Netzperfekt\SaasDeadman\Models\DMSwitch;

class ConfirmationController extends Controller
{
    public function confirmSwitch(Request $request, int $switchId, int $contactId, string $confirmHash = '')
    {
        if (! $request->hasValidSignature()) {
            abort(401);
        }

        $switch = DMSwitch::where('id', $switchId)->first();

        if (empty($confirmHash))
        {
            return $this->showRequest($switch, $contactId);
        }

        $firstUser = $switch->team->members->first();

        if($switch->isTriggered())
        {
            return view('saas-deadman::frontend.message', [
                'message' => __('deadman-frontend-already-triggered', [
                    'date' => ($switch->triggered_at?->formatWithTimezone('d.m.Y H:i', $firstUser->timezone) ?? '---')
                ])
            ]);
        }

        elseif($switch->isCompletelyConfirmed())
        {
            return view('saas-deadman::frontend.message', [
                'message' => __('deadman-frontend-already-confirmed-until', [
                    'date' => ($switch->next_confirmation?->formatWithTimezone('d.m.Y H:i', $firstUser->timezone) ?? '---')
                ])
            ]);
        }

        elseif($switch->isConfirmedForContact($contactId))
        {
            return view('saas-deadman::frontend.message', [
                'message' => __('deadman-frontend-already-confirmed')
            ]);
        }

        $switch->confirmFromContact($contactId);

        if($switch->isCompletelyConfirmed())
        {
            $nextConfirmationDate = SwitchService::setConfirmed($switch);
            $message =  __('deadman-frontend-confirmed-until', [
                'date' => $nextConfirmationDate->formatWithTimezone('d.m.Y H:i', $firstUser->timezone)
            ]);
        }
        else
        {
            $message = __('deadman-frontend-confirmed-waiting-for-others');
        }

        activity()
            ->performedOn($switch)
            ->withProperties([
                'contact' => $contactId
            ])
            ->log(ActivityLogType::SwitchConfirmed->value);

        return view('saas-deadman::frontend.message', [
            'message' => $message
        ]);
    }

    private function showRequest(DMSwitch $switch, int $contactId)
    {
        $url = URL::temporarySignedRoute(
            'front.switch.confirm',
            now()->addHours(config('saas-deadman.switch_confirmation_validity')),
            [
                'switchId' => $switch->id,
                'contactId' => $contactId,
                'confirmhash' => Str::random(16)
            ]
        );

        $message = __('deadman-frontend-confirm-now', ['name' => $switch->title]);
        $messageUrl = __('deadman-frontend-confirm-now-click-url');

        activity()
            ->performedOn($switch)
            ->withProperties([
                'contact' => $contactId
            ])
            ->log(ActivityLogType::SwitchConfirmationPre->value);

        return view('saas-deadman::frontend.message', [
            'message' => $message,
            'url' => $url,
            'messageUrl' => $messageUrl
        ]);
    }
}
