<?php

namespace Netzperfekt\SaasDeadman\Controllers;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasDeadman\Enums\ActivityLogType;
use Netzperfekt\SaasDeadman\Models\Action;
use Netzperfekt\SaasDeadman\Models\Channel;
use Netzperfekt\SaasDeadman\Models\Contact;
use Netzperfekt\SaasDeadman\Models\DMSwitch;
use Netzperfekt\SaasDeadman\States\SwitchTriggered;

class NotificationController extends Controller
{
    public function test(Request $request, int $channelId, int $switchId)
    {
        if (!$request->hasValidSignature()) {
            abort(401);
        }

        $channel = Channel::find($channelId);
        $switch = DMSwitch::find($switchId);

        return view('saas-deadman::frontend.message', [
            'message' => __('deadman-mail-test-message', [
                'channel' => $channel->title, 'switch' => $switch->title
            ])
        ]);
    }

    public function verify(Request $request,
                           int $channelId,
                           int $contactId,
                           string $recipient,
                           string $confirmHash = '')
    {
        if (!$request->hasValidSignature()) {
            abort(401);
        }

        if (empty($confirmHash))
        {
            return $this->verifyNow($request, $channelId, $contactId, $recipient);
        }

        $contact = Contact::find($contactId);
        $channel = Channel::find($channelId);
        $code = $request->get('code');

        foreach($contact->recipients($channel)->get() as $thisRecipient)
        {
            if(in_array($recipient, $thisRecipient->recipients))
            {
                if( ! array_key_exists($recipient, $thisRecipient->confirmations))
                {
                    abort(404);
                }

                if ($thisRecipient->confirmations[$recipient]['code'] == '' ||
                    $thisRecipient->confirmations[$recipient]['code'] != $code)
                {
                    return view('saas-deadman::frontend.message', [
                        'message' => __('deadman-recipient-verify-wrongcode')
                    ]);
                }

                $confirmations = $thisRecipient->confirmations;
                $confirmations[$recipient] = [
                    'code'      => '',
                    'confirmed' => true,
                    'date'      => Carbon::now()
                ];

                $thisRecipient->confirmations = $confirmations;
                $thisRecipient->save();
            }
        }

        activity()
            ->performedOn($contact)
            ->withProperties([
                'channel'   => $channelId,
                'contact'   => $contactId,
                'recipient' => $recipient,
                'ip'        => $request->getClientIp(),
                'useragent' => $request->userAgent()
            ])
            ->log(ActivityLogType::VerifyRecipients->value);

        return view('saas-deadman::frontend.message', [
            'message' => __('deadman-recipient-verified', ['recipient' => $recipient])
        ]);
    }

    private function verifyNow(Request $request,
                               int $channelId,
                               int $contactId,
                               string $recipient)
    {
        $contact = Contact::find($contactId);

        $url = URL::signedRoute(
            'front.mail.verify', [
                'channelId'   => $channelId,
                'contactId'   => $contactId,
                'recipient'   => $recipient,
                'confirmhash' => Str::random(16)
            ]
        );

        $message = __('deadman-frontend-verify-now', ['recipient' => $recipient]);
        $messageInfo = __('deadman-frontend-verify-now-info');

        $messageUrl = __('deadman-frontend-verify-now-click-url');

        activity()
            ->performedOn($contact)
            ->withProperties([
                'recipient' => $recipient
            ])
            ->log(ActivityLogType::VerifyRecipientsPre->value);

        return view('saas-deadman::frontend.verify', [
            'message' => $message,
            'messageInfo' => $messageInfo,
            'url' => $url,
            'messageUrl' => $messageUrl
        ]);
    }


    public function showNotification(Request $request,
                                     int $switchId,
                                     int $actionId,
                                     int $contactId,
                                     string $confirmHash = '')
    {
        if (! $request->hasValidSignature()) {
            abort(401);
        }

        $switch = DMSwitch::where('id', $switchId)->firstOrFail();
        if( ! $switch->state->equals(SwitchTriggered::class))
        {
            activity()
                ->performedOn($switch)
                ->withProperties([
                    'action'    => $actionId,
                    'contact'   => $contactId,
                    'ip'        => $request->getClientIp(),
                    'useragent' => $request->userAgent()
                ])
                ->log(ActivityLogType::ShowNotificationWithdrawn->value);

            return view('saas-deadman::frontend.message', [
                'message' => __('deadman-notification-not-available')
            ]);
        }

        $notification = Action::where('id', $actionId)->firstOrFail();
        if (empty($confirmHash))
        {
            return $this->showRequest($switch, $actionId, $contactId);
        }

        $action = $switch->load('actions')->actions->filter(fn($a) => $a->id == $notification->id)->first();

        $teamId = $notification->team->id;
        $team = Team::where('id', $teamId)->first();
        $firstUserOfThisTeam = $team->members()->first();
        $firstUserOfThisTeam->current_team_id = $teamId; // can be overwritten - is newly set at next login of this user
        $firstUserOfThisTeam->save();

        activity()->withoutLogs(fn () =>
            Auth::login($firstUserOfThisTeam)
        );

        activity()
            ->performedOn($switch)
            ->withProperties([
                'action'    => $actionId,
                'contact'   => $contactId,
                'ip'        => $request->getClientIp(),
                'useragent' => $request->userAgent()
            ])
            ->log(ActivityLogType::ShowNotification->value);

        return view('saas-deadman::frontend.notification', [
            'sender' => $firstUserOfThisTeam,
            'team'   => $team,
            'switch' => $switch,
            'action' => $action
        ]);
    }

    private function showRequest(DMSwitch $switch, int $actionId, int $contactId)
    {
        $url = URL::signedRoute(
            'front.notification.show', [
                'switchId' => $switch->id,
                'actionId' => $actionId,
                'contactId' => $contactId,
                'confirmhash' => Str::random(16)
            ]
        );

        $message = __('deadman-frontend-notification-shownow', ['name' => $switch->title]);
        $messageInfo = __('deadman-frontend-notification-shownow-info');

        $messageUrl = __('deadman-frontend-notification-shownow-click-url');

        activity()
            ->performedOn($switch)
            ->withProperties([
                'action' => $actionId
            ])
            ->log(ActivityLogType::ShowNotificationPre->value);

        return view('saas-deadman::frontend.message', [
            'message' => $message,
            'messageInfo' => $messageInfo,
            'url' => $url,
            'messageUrl' => $messageUrl
        ]);
    }
}
