<?php

declare(strict_types=1);

namespace Netzperfekt\SaasDeadman\Encryption;

use Aws\S3\S3Client;
use Illuminate\Encryption\Encrypter;
use Illuminate\Filesystem\AwsS3V3Adapter;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Contracts\Foundation\Application;
use League\Flysystem\AwsS3V3\AwsS3V3Adapter as S3Adapter;
use League\Flysystem\AwsS3V3\PortableVisibilityConverter;
use League\Flysystem\Filesystem as Flysystem;
use League\Flysystem\FilesystemAdapter as FlysystemAdapter;
use League\Flysystem\Visibility;

class EncryptedS3Adapter
{
    public static function make(Application $app, array $config): AwsS3V3Adapter
    {
        $root = (string) ($config['root'] ?? '');

        $visibility = new PortableVisibilityConverter(
            $config['visibility'] ?? Visibility::PUBLIC
        );

        $streamReads = $config['stream_reads'] ?? false;

        $client = new S3Client([
            'version'      => 'latest',
            'credentials'  => [
                'key'    => $config['key'],
                'secret' => $config['secret'],
            ],
            'key'          => $config['key'],
            'secret'       => $config['secret'],
            'region'       => $config['region'],
            'bucket'       => $config['bucket'],
            'cdn_endpoint' => $config['cdn_endpoint'],
            'url'          => $config['url'],
            'endpoint'     => $config['endpoint'],
            'use_path_style_endpoint' => $config['use_path_style_endpoint'],
            'throw'        => $config['throw'],
        ]);

        $s3adapter = new S3Adapter(
            $client,
            $config['bucket'],
            $root,
            $visibility,
            null,
            $config['options'] ?? [],
            $streamReads
        );

        $teamId = Auth::user()->getCurrentTeamId();

        $encryptionKey = Encryption::getEncryptionKeyFor($teamId);
        $encrypter = new Encrypter($encryptionKey, 'aes-256-gcm');
        $adapter = new EncryptedFilesystemAdapter($s3adapter, $encrypter);

        return new AwsS3V3Adapter(self::createFlysystem($adapter, $config),
            $s3adapter, $config, $client
        );
    }

    private static function createFlysystem(FlysystemAdapter $adapter, array $config)
    {
        return new Flysystem($adapter, Arr::only($config, [
            'directory_visibility',
            'disable_asserts',
            'retain_visibility',
            'temporary_url',
            'url',
            'visibility',
        ]));
    }
}
