<?php

namespace Netzperfekt\SaasDeadman\Filament\Resources\ActionResource;

use Filament\Forms\Components\Builder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Guava\FilamentKnowledgeBase\Contracts\HasKnowledgeBase;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasDeadman\Enums\ActionType;
use Netzperfekt\SaasDeadman\Filament\Resources\ActionResource\Pages\CreateAction;
use Netzperfekt\SaasDeadman\Filament\Resources\ActionResource\Pages\EditAction;
use Netzperfekt\SaasDeadman\Filament\Resources\ActionResource\Pages\ListActions;
use Netzperfekt\SaasDeadman\Filament\Resources\ContentBlocks;
use Netzperfekt\SaasDeadman\Models\Action;
use Filament\Forms\Components\Actions\Action as FilamentAction;

class ActionResource extends Resource implements HasKnowledgeBase
{
    protected static ?string $model = Action::class;

    protected static ?string $navigationIcon = 'far-envelope';
    protected static ?string $recordTitleAttribute = 'subject';
    protected static ?int $navigationSort = -1;

    public const MAX_RECIPIENTS = 10;

    public static function getNavigationGroup(): ?string
    {
        return __('deadman-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return __('deadman-action');
    }

    public static function getPluralModelLabel(): string
    {
        return __('deadman-actions');
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Typ' => $record->type->getLabel()
        ];
    }

    public static function form(Form $form): Form
    {
        return self::getForm($form);
    }

    public static function getForm(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Split::make([
                    Section::make()
                        ->schema([
                            TextInput::make('subject')
                                ->label(fn($get) => $get('type') == ActionType::Notification->value ? __('deadman-subject') : __('deadman-title'))
                                ->columnSpan(3)
                                ->required()
                                ->maxLength(255)
                            ->maxWidth('sm'),

                            Toggle::make('attach_contents')
                                ->label(__('deadman-attach-contents'))
                                ->inline(false)
                                ->hintIcon(fn($get) => $get('attach_contents') ? 'fas-triangle-exclamation' : '')
                                ->hintIconTooltip(fn($get) => $get('attach_contents') ? __('deadman-attach-contents-hint') : '')
                                ->hintColor('danger')
                                ->visible(fn($get) => $get('type') == ActionType::Notification->value)
                                ->columnStart(4)
                                ->columnSpan(2)
                                ->live()
                        ])
                        ->grow(),

                    Section::make([
                        Select::make('type')
                            ->label(__('deadman-type'))
                            ->options(ActionType::class)
                            ->default(ActionType::Notification->value)
                            ->required()
                            ->live(),

                        /* TODO später
                        Toggle::make('encrypted')
                            ->label(__('deadman-encrypted'))
                            ->helperText(new HtmlString(__('deadman-encrypted-helper')))
                            ->live()
                            ->default(false)
                            ->onColor(Color::Orange)
                            ->onIcon('fas-shield-heart')
                            ->maxWidth('xs')
                        */
                    ])
                        ->grow(false)
                ])->from('xl'),

                Builder::make('contents')
                    ->visible(fn($get) => $get('type') == ActionType::Notification->value)
                    ->label(fn($get) => new HtmlString(
                        '<b>' . __('deadman-notification-contents') . '</b>' . ($get('encrypted')
                            ? ' | <b>' . __('deadman-is-encrypted') . '</b>'
                            : ''))
                    )
                    ->blocks(ContentBlocks::getContentBlocks())
                    ->blockPickerWidth(MaxWidth::ExtraSmall)
                    ->reorderable(true)
                    ->collapsible()
                    ->blockNumbers(false)
                    ->addActionLabel(__('deadman-content-add-entry'))
                    ->deleteAction(fn (FilamentAction $action) => $action->requiresConfirmation())
                    ->extraAttributes(fn($get) => $get('encrypted')
                        ? ['style' => 'border: 10px solid #ffedd5; border-radius: 20px', 'class' => 'block-picker-add-new']
                        : ['class' => 'action-blocks'])
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('subject')
                    ->label(__('deadman-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('contents')
                    ->label(__('deadman-contents'))
                    ->badge()
                    ->limitList(5)
                    ->formatStateUsing(function($state) {
                        return __('deadman-content-' . $state['type']);
                    }),

                /* TODO später
                Tables\Columns\ToggleColumn::make('encrypted')
                    ->label(__('deadman-encrypted'))
                    ->onColor(Color::Orange)
                    ->onIcon('fas-shield-heart')
                    ->disabled(),
                */

                Tables\Columns\TextColumn::make('updated_at')
                    ->date('d.m.Y')
                    ->label(__('deadman-last-update')),

                Tables\Columns\TextColumn::make('type')
                    ->label(__('deadman-type'))
                    ->badge()
            ])

            ->recordClasses(fn(Action $action) => $action->encrypted ? 'bg-orange-100 dark:bg-gray-600' : '')

            ->filters([
                Tables\Filters\TrashedFilter::make()
            ])

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()
                    ->tooltip('Bearbeiten'),

                Tables\Actions\DeleteAction::make()->iconButton()
                    ->tooltip('Löschen'),

                Tables\Actions\ForceDeleteAction::make()->iconButton()
                    ->tooltip('Endgültig löschen'),

                Tables\Actions\RestoreAction::make()->iconButton()
                    ->tooltip('Wiederherstellen')
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])

            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ])

            ->defaultSort('subject');
    }

    public static function getRelations(): array
    {
        return [
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListActions::route('/'),
            'create' => CreateAction::route('/create'),
            'edit' => EditAction::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    public static function getDocumentation(): array
    {
        return [
            'actions'
        ];
    }
}
