<?php

namespace Netzperfekt\SaasDeadman\Filament\Resources\ContactResource;

use Awcodes\TableRepeater\Components\TableRepeater;
use Awcodes\TableRepeater\Header;
use Carbon\Carbon;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconPosition;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Guava\FilamentKnowledgeBase\Contracts\HasKnowledgeBase;
use Illuminate\Contracts\Support\Htmlable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasBase\SaasBase;
use Netzperfekt\SaasDeadman\Filament\Resources\ContactResource\RelationManagers\NotificationTargetsRelationManager;
use Netzperfekt\SaasDeadman\Enums\ConfirmationFrequency;
use Netzperfekt\SaasDeadman\Enums\ContactType;
use Netzperfekt\SaasDeadman\Filament\Resources\ContactResource\Pages\CreateContact;
use Netzperfekt\SaasDeadman\Filament\Resources\ContactResource\Pages\EditContact;
use Netzperfekt\SaasDeadman\Filament\Resources\ContactResource\Pages\ListContacts;
use Netzperfekt\SaasDeadman\Filament\Resources\ContactResource\RelationManagers\SwitchRelationManager;
use Netzperfekt\SaasDeadman\Models\Channel;
use Netzperfekt\SaasDeadman\Models\Contact;
use Netzperfekt\SaasDeadman\Services\SwitchService;

class ContactResource extends Resource implements HasKnowledgeBase
{
    protected static ?string $model = Contact::class;

    protected static ?string $navigationIcon = 'heroicon-o-building-office';
    protected static ?string $recordTitleAttribute = 'lastname';
    protected static ?int $navigationSort = -2;

    private const MAX_RECIPIENTS = 5;

    public static function getNavigationGroup(): ?string
    {
        return __('deadman-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return __('deadman-contact');
    }

    public static function getPluralModelLabel(): string
    {
        return __('deadman-contacts');
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['lastname', 'firstname', 'company'];
    }

    public static function getGlobalSearchResultTitle(Model $record): string | Htmlable
    {
        return new HtmlString($record->fullname . '<br>' . $record->company);
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Typ' => $record->type->getLabel()
        ];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Split::make([
                    Section::make()
                        ->compact()
                        ->schema([
                            Section::make('')
                                ->compact()
                                ->columns(3)
                                ->schema([
                                    TextInput::make('lastname')
                                        ->label(__('deadman-lastname'))
                                        ->extraAttributes(['class' => 'font-semibold'])
                                        ->required()
                                        ->maxLength(255)
                                        ->autofocus(),

                                    TextInput::make('firstname')
                                        ->label(__('deadman-firstname'))
                                        ->extraAttributes(['class' => 'font-semibold'])
                                        ->required()
                                        ->maxLength(255)
                                        ->autofocus(),

                                    TextInput::make('company')
                                        ->label(__('deadman-company'))
                                        ->maxLength(255),

                                    Grid::make()
                                        ->columns(4)
                                        ->schema([
                                        TextInput::make('street')
                                            ->label(__('deadman-street'))
                                            ->maxLength(255),

                                        TextInput::make('zip')
                                            ->label(__('deadman-zip'))
                                            ->maxLength(255),

                                        TextInput::make('city')
                                            ->label(__('deadman-city'))
                                            ->maxLength(255),

                                        Select::make('country')
                                            ->label(__('deadman-country'))
                                            ->options(SaasBase::getCountryOptions())
                                            ->required(),
                                    ])
                                ]),

                            TableRepeater::make('channel_recipients')
                                ->label(__('deadman-channel-contacts'))
                                ->addActionLabel(__('deadman-channel-add-contact'))
                                ->columnSpanFull()
                                ->relationship('recipients')
                                ->renderHeader(false)
                                ->headers([
                                    Header::make('channel_id'),
                                    Header::make('recipients'),
                                    Header::make('confirmations')
                                ])
                                ->schema([
                                    Select::make('channel_id')
                                        ->label('')
                                        ->placeholder(__('deadman-select-channel'))
                                        ->options(function()
                                        {
                                            $channels = [];
                                            foreach(Channel::all() as $channel)
                                            {
                                                $title = $channel->title;
                                                if(! $channel->active)
                                                {
                                                    $title .= ' (inaktiv)';
                                                }
                                                $channels[$channel->id] = $title;
                                            }

                                            return $channels;
                                        })
                                        ->maxWidth('175px')
                                        ->extraAttributes(['class' => 'font-semibold'])
                                        ->required()
                                        ->default(Channel::where('type', Channel::DefaultChannel)->first()->id),


                                    TagsInput::make('recipients')
                                        ->label(__('deadman-recipients'))
                                        ->placeholder(__('deadman-channel-contacts-recipient'))
                                        ->required()
                                        ->rules([
                                            function ($get)
                                            {
                                                return function (string $attribute, $value, \Closure $fail) use ($get)
                                                {
                                                    if(count($value) > self::MAX_RECIPIENTS)
                                                    {
                                                        $fail(__('Max. :count recipients allowed.', [
                                                            'count' => self::MAX_RECIPIENTS
                                                        ]));
                                                    }

                                                    $channel = Channel::find($get('channel_id'));
                                                    if ( ! $channel)
                                                    {
                                                        return '';
                                                    }

                                                    $validationRule = $channel->type->getValidationRule();
                                                    $validator = \Validator::make(['recipients' => $value], [
                                                        'recipients.*' => [$validationRule]
                                                    ], [
                                                        'recipients.*.email' => __('":input" is not an email'),
                                                        'recipients.*.regex' => __('":input" is not valid')
                                                    ]);

                                                    if( ! $validator->valid()) {
                                                        $fail(implode(', ', $validator->errors()->all()));
                                                    }

                                                    return '';
                                                };
                                            }
                                        ]),

                                    Placeholder::make('confirmations')
                                        ->label('')
                                        ->content(function ($get)
                                        {
                                            $s = '<small><ul>';
                                            foreach($get('recipients') as $recipient)
                                            {
                                                $confirmations = $get('confirmations') ?? [];
                                                if(array_key_exists($recipient, $confirmations))
                                                {
                                                    $confirmation = $get('confirmations')[$recipient];
                                                    $s .= '<li>';
                                                    if( ! $confirmation['confirmed'])
                                                    {
                                                        $tooltip = __('deadman-recipient-waiting-for-confirmation');
                                                        $s .= "<s x-tooltip=\"{ content: '" . $tooltip . "' }\">" .
                                                            $recipient .
                                                        "</s>";
                                                    }
                                                    else
                                                    {
                                                        $tooltip = __('deadman-recipient-confirmed-on',
                                                            ['date' =>
                                                                Carbon::make($confirmation['date'])
                                                                ->formatWithTimezone('d.m.Y H:i')]
                                                        );
                                                        $s .= "<span class=\"text-green-700\" x-tooltip=\"{ content: '" . $tooltip . "' }\">" .
                                                            $recipient .
                                                        "</span>";
                                                    }
                                                    $s .= '</li>';
                                                }
                                            }
                                            $s .= '</ul></small>';

                                            return new HtmlString($s);
                                        })
                                ])
                                ->default(fn() => self::addDefaultChannels())
                                ->deleteAction(fn (Action $action) => $action->requiresConfirmation())
                                ->reorderable(false)
                                ->extraItemActions([
                                    fn($record) => Action::make('verifyRecipients')
                                        ->label('')
                                        ->icon('fas-person-circle-question')
                                        ->tooltip(__('deadman-recipients-verify'))
                                        ->action(function (array $arguments, TableRepeater $component) use($record): void
                                        {
                                            $itemData = $component->getItemState($arguments['item']);
                                            $channel = Channel::where('id', $itemData['channel_id'])->first();

                                            $count = SwitchService::sendVerificationNotifications($record, $channel);

                                            if($count > 0)
                                            {
                                                Notification::make()
                                                    ->title(__('deadman-notification-sent', ['count' => $count]))
                                                    ->success()
                                                    ->send();
                                            }
                                            else {
                                                Notification::make()
                                                    ->title(__('deadman-notification-notsent'))
                                                    ->danger()
                                                    ->send();
                                            }
                                        })
                                        ->requiresConfirmation(fn() => self::getLabel())
                                        ->modalHeading(function($arguments, $component)
                                        {
                                            try {
                                                $itemData = $component->getItemState($arguments['item']);
                                                $channel = Channel::where('id', $itemData['channel_id'])->first();

                                                return new HtmlString(__('deadman-recipients-verify-now', ['channel' => $channel->title]));
                                            }
                                            catch(\Exception)
                                            {
                                                return '';
                                            }
                                        })
                                        ->modalWidth(MaxWidth::Medium)
                                ])
                        ]),

                    Section::make()
                        ->compact()
                        ->schema([
                            Select::make('type')
                                ->label(__('deadman-type'))
                                ->required()
                                ->options(ContactType::class)
                                ->live()
                                ->rules([
                                    function ($get)
                                    {
                                        return function (string $attribute, $value, \Closure $fail) use ($get)
                                        {
                                            if($value == ContactType::Myself->value)
                                            {
                                                $recordId = $get('id') ?? 0;
                                                $contactsWithSameType = Contact
                                                    ::onlyMyRecords()
                                                    ->where('type', $value)
                                                    ->where('id', '!=', $recordId)
                                                    ->get();

                                                if ($contactsWithSameType->isNotEmpty()) {
                                                    $fail(__('deadman-contact-only-one-own'));
                                                }
                                            }
                                        };
                                    }
                                ]),

                            Toggle::make('must_confirm')
                                ->label(__('deadman-contact-must-confirm'))
                                ->live()
                                ->visible(fn($get) => $get('type') != ContactType::Myself->value),

                            Select::make('confirmation_frequency')
                                ->label(__('deadman-confirmation-frequency'))
                                ->required()
                                ->options(ConfirmationFrequency::class)
                                ->live()
                                ->visible(fn($get) =>
                                    $get('type') != ContactType::Myself->value &&
                                    $get('must_confirm')),

                            DateTimePicker::make('last_confirmation')
                                ->label(__('deadman-last-confirmation'))
                                ->disabled(true)
                                ->visible(fn($get) =>
                                    $get('type') != ContactType::Myself->value &&
                                    $get('must_confirm')),

                            DateTimePicker::make('next_confirmation')
                                ->label(__('deadman-next-confirmation'))
                                ->disabled(true)
                                ->visible(fn($get) =>
                                    $get('type') != ContactType::Myself->value &&
                                    $get('must_confirm') &&
                                    $get('confirmation_frequency') != ConfirmationFrequency::Once->value),

                        ])->grow(false)
                ])->from('xl'),

                Section::make(__('deadman-comments'))
                    ->schema([
                        Textarea::make('comments')
                            ->label('')
                            ->rows(5)
                    ])
                    ->compact()
                    ->collapsible()
                    ->persistCollapsed()
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultPaginationPageOption(25)
            ->groups([
                Tables\Grouping\Group::make('type')
                    ->label(__('deadman-type'))
                    ->collapsible()
            ])

            ->columns([
                Tables\Columns\TextColumn::make('lastname')
                    ->label(__('deadman-name'))
                    ->formatStateUsing(function (Contact $contact)
                    {
                        return $contact->lastname . ', ' . $contact->firstname;
                    })
                    ->weight(FontWeight::SemiBold)
                    ->searchable()
                    ->tooltip(function ($record)
                    {
                        if($record->isOwnContact()) {
                            return __('deadman-own-contact');
                        }

                        return '';
                    })
                    ->icon(function ($record)
                    {
                        if($record->isOwnContact()) {
                            return 'fas-person-burst';
                        }

                        return '';
                    })
                    ->iconColor('warning')
                    ->iconPosition(IconPosition::After),


                Tables\Columns\TextColumn::make('company')
                    ->label(__('deadman-company'))
                    ->searchable(),

                Tables\Columns\TextColumn::make('city')
                    ->label(__('deadman-city'))
                    ->searchable(),

                Tables\Columns\TextColumn::make('type')
                    ->label(__('deadman-type'))
                    ->badge()
                    ->color('warning'),

                Tables\Columns\ToggleColumn::make('must_confirm')
                    ->disabled(true)
                    ->label(__('deadman-contact-must-confirm-col')),

                Tables\Columns\TextColumn::make('last_confirmation')
                    ->disabled(true)
                    ->label(__('deadman-last-confirmation'))
                    ->formatStateUsing(fn($record, $state) => $record->must_confirm ? $state->formatWithTimezone('d.m.y H:i') : ''),

                Tables\Columns\TextColumn::make('recipients_count')
                    ->counts('recipients')
                    ->label(__('deadman-recipients-count'))
                    ->badge(),

                /* TODO raus? braucht nur platz und gibt wenig zusatzinfos
                Tables\Columns\TextColumn::make('switches_count')
                    ->counts('switches')
                    ->label(__('deadman-switches-count'))
                    ->formatStateUsing(fn ($state, Contact $contact) => $contact?->type == ContactType::Myself)
                    ->badge(fn($record) => $record?->type == ContactType::Myself),
                */
            ])

            ->recordClasses(fn($record) => $record->type == ContactType::Myself ? 'bg-blue-50 dark:bg-gray-600' : '')
            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('type')
                    ->label(__('deadman-type'))
                    ->options(ContactType::class)
            ])
            
            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),
                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])
            
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ])
            
            ->defaultSort(function (Builder $query): Builder {
                return $query
                    ->orderByRaw('IF(type = ' . ContactType::Myself->value . ', -1, lastname)');
            });
    }

    public static function getRelations(): array
    {
        return [
            SwitchRelationManager::class,
            NotificationTargetsRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListContacts::route('/'),
            'create' => CreateContact::route('/create'),
            'edit' => EditContact::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    public static function addDefaultChannels(): array
    {
        $allChannels = Channel::active()->get();

        $result = [];
        foreach($allChannels as $channel)
        {
            array_push($result, [
                'channel_id'    => $channel->id,
                'recipients'    => [],
                'confirmations' => []
            ]);
        }

        return $result;
    }

    public static function getDocumentation(): array
    {
        return [
            'contacts'
        ];
    }
}
