<?php

namespace Netzperfekt\SaasDeadman\Filament\Resources;

use Awcodes\TableRepeater\Components\TableRepeater;
use Awcodes\TableRepeater\Header;
use Filament\Facades\Filament;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\Builder\Block;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Illuminate\Support\Facades\URL;
use Netzperfekt\SaasDeadman\Filament\Components\FileDownload;

class ContentBlocks
{
    public const TYPE_MESSAGE_BLOCK    = 'message-block';
    public const TYPE_CHECKLIST_BLOCK  = 'checklist-block';
    public const TYPE_LINK_BLOCK       = 'link-block';
    public const TYPE_BIZCAT_BLOCK     = 'bizcat-block';
    public const TYPE_EMERGENCY_FOLDER = 'emergency-folder-block';
    public const TYPE_ATTACHMENT_BLOCK = 'attachment-block';

    public static function getContentBlocks()
    {
        return [
            self::getContentBlockMessage(),
            self::getContentBlockChecklist(),
            self::getContentBlockLink(),
            self::getContentBlockEmergencyFolder(),
            self::getAttachmentBlockLink(),
            self::getBizCatBlock()
        ];
    }

    public static function getContentBlockContentsAsHtml(string $blockType, array $data, int $teamId) : string
    {
        $s = '';
        if($blockType == self::TYPE_MESSAGE_BLOCK)
        {
            $s = '<div class="content-title">' . __('deadman-content-message') . ':</div>' .
                 '<div class="content-field">' . $data['message'] . '</div>';
        }

        elseif($blockType == self::TYPE_CHECKLIST_BLOCK)
        {
            $s  = '<div class="content-title">' . $data['title'] . ':</div>';
            $s .= '<div class="content-field"><ul>';
            foreach($data['checklist_entry'] as $entry)
            {
                $s .= '<li>' . $entry['title'];
                if(! empty($entry['additional']))
                {
                    $s .= '<br><small>' . $entry['additional'] . '</small>';
                }
                $s .= '</li>';
            }
            $s .= '</ul></div>';
        }

        elseif($blockType == self::TYPE_LINK_BLOCK)
        {
            $link = $data['link'];
            $s  = '<div class="content-title">' . __('deadman-notification-link') . '</div>';
            $s .= '<div class="content-field">';
            $s .= '<a href="' . $link . '" target="_blank">' . $link . '</a>';
            $s .= '</div>';
        }

        elseif($blockType == self::TYPE_BIZCAT_BLOCK)
        {
            $link = $data['bizcat-link'];
            $title = $data['title'] ?? __('deadman-bizcat-infos');

            $s  = '<div class="content-title">' . __('deadman-notification-bizcat-title') . '</div>';
            $s .= '<div class="content-field">';
            $s .= ' <a href="' . $link . '" target="_blank">' . $title . '</a>';
            $s .= '</div>';
        }

        elseif($blockType == self::TYPE_EMERGENCY_FOLDER)
        {
            $s  = '<div class="content-title">'  . __('deadman-emergency-folder') . '<b>' . $data['title'] . '</b></div>';
            $s .= '<div class="content-field">...TODO...</div>';
            // TODO
        }

        elseif($blockType == self::TYPE_ATTACHMENT_BLOCK)
        {
            $title = $data['title'];
            $password = $data['password'];
            $downloadLink = URL::signedRoute('front.attachment.download', [
                'teamId'   => $teamId,
                'disk'     => config('saas-deadman.attachments_disk'),
                'filename' => urlencode(urlencode($data['attachment']))
            ]);

            $s  = '<div class="content-title">' . __('deadman-notification-attachment') . '</div>';
            $s .= '<div class="content-field">';
            $s .= '<a href="' . $downloadLink . '" target="_blank">' . $title . '</a>';
            if(! empty($password))
            {
                $s .= ' - <b>' . __('deadman-notification-password') . '</b>: ' . $password;
            }
            $s .= '</div>';
            $s .= '</div>';
        }

        else {
            $s = '*** unknown content block type: ' . $blockType;
        }

        return $s;
    }

    private static function getContentBlockMessage(): Block
    {
        return Block::make(self::TYPE_MESSAGE_BLOCK)
            ->htmlLabel(__('deadman-content-message-block'), 'font-bold text-orange-600')
            ->icon('far-envelope')
            ->maxItems(1)
            ->schema([
                Textarea::make('message')
                    ->label(__('deadman-content-message'))
                    ->rows(5)
                    ->required()
            ]);
    }

    private static function getContentBlockChecklist(): Block
    {
        return Block::make(self::TYPE_CHECKLIST_BLOCK)
            ->htmlLabel(__('deadman-content-checklist-block'), 'font-bold text-orange-600')
            ->icon('fas-list-check')
            ->schema([
                TextInput::make('title')
                    ->label(__('deadman-content-title'))
                    ->extraAttributes(['class' => 'font-semibold'])
                    ->required(),

                TableRepeater::make('checklist_entry')
                    ->label('')
                    ->columnSpanFull()
                    ->renderHeader(false)
                    ->headers([
                        Header::make('title'),
                        Header::make('additional')
                    ])
                    ->schema([
                        TextInput::make('title')
                            ->label('')
                            ->maxWidth('250px')
                            ->required()
                            ->placeholder(__('deadman-checklist-entry-title')),

                        TextInput::make('additional')
                            ->placeholder(__('deadman-checklist-entry-additional'))
                    ])
                    ->addActionLabel(__('deadman-checklist-add-entry'))
            ]);
    }

    private static function getContentBlockLink(): Block
    {
        return Block::make(self::TYPE_LINK_BLOCK)
            ->htmlLabel(__('deadman-content-link-block'), 'font-bold text-orange-600')
            ->icon('fas-link')
            ->schema([
                Grid::make(4)
                    ->schema([
                        TextInput::make('link')
                            ->label(__('deadman-content-link'))
                            ->columnSpan(2)
                            ->url()
                            ->required(),

                        TextInput::make('username')
                            ->label(__('deadman-content-username'))
                            ->password()
                            ->revealable()
                            ->helperText(__('deadman-content-optional-protection')),

                        TextInput::make('password')
                            ->label(__('deadman-content-password'))
                            ->password()
                            ->revealable()
                            ->helperText(__('deadman-content-optional-protection')),
                    ])
            ]);
    }

    private static function getBizCatBlock(): Block
    {
        return Block::make(self::TYPE_BIZCAT_BLOCK)
            ->htmlLabel(__('deadman-content-bizcat-block'), 'font-bold text-orange-600')
            ->icon('fas-arrows-to-eye')
            ->schema([
                Grid::make(4)
                    ->schema([
                        TextInput::make('title')
                            ->label(__('deadman-title'))
                            ->placeholder(__('deadman-checklist-entry-title'))
                            ->columnSpan(1)
                            ->maxWidth('250px')
                            ->required(),

                        TextInput::make('bizcat-link')
                            ->label(__('deadman-bizcat-link'))
                            ->hint(__('deadman-bizcat-link-hint'))
                            ->columnSpan(3)
                            ->url()
                            ->startsWith(['https://bizcat.test', 'https://bizcat.eu'])
                            ->required()
                            ->live()
                            ->suffixAction(Action::make('open-bizcat-link')
                                ->icon('fas-link')
                                ->url(fn($get) => $get('bizcat-link'))
                                ->disabled(fn($get) => $get('bizcat-link') == '')
                                ->openUrlInNewTab())
                    ])
            ]);
    }

    private static function getContentBlockEmergencyFolder(): Block
    {
        return Block::make(self::TYPE_EMERGENCY_FOLDER)
            ->htmlLabel(__('deadman-content-emergency-folder-block'), 'font-bold text-orange-600')
            ->icon('far-folder')
            ->maxItems(1)
            ->schema([
                TextInput::make('title')
                    ->label(__('deadman-content-title'))
                    ->extraAttributes(['class' => 'font-semibold'])
                    ->required(),

                // TODO inhalte/blöcke noch aufbauen, hier auch dateiuploads erlauben
                // TODO analog zum "Notfallordner" in ~/Downloads/Notfallordner

            ]);
    }

    private static function getAttachmentBlockLink(): Block
    {
        return Block::make(self::TYPE_ATTACHMENT_BLOCK)
            ->htmlLabel(__('deadman-content-attachment-block'), 'font-bold text-orange-600')
            ->icon('far-file')
            ->schema([
                Grid::make(6)
                    ->schema([
                        TextInput::make('title')
                            ->label(__('deadman-content-title'))
                            ->required()
                            ->columnSpan(3),

                        TextInput::make('password')
                            ->label(__('deadman-content-password'))
                            ->password()
                            ->revealable()
                            ->columnSpan(1)
                            ->helperText(__('deadman-content-optional-password'))
                            ->hintIcon('fas-info-circle')
                            ->hintIconTooltip(__('deadman-content-optional-password-hint')),

                        FileUpload::make('attachment')
                            ->directory('attachments/t' . Filament::getTenant()->id)
                            ->required()
                            ->disk(config('saas-deadman.attachments_disk'))
                            ->maxSize(config('saas-deadman.attachments_max_size'))
                            ->visibility('private')
                            ->moveFiles()
                            ->downloadable(false)
                            ->openable(false)
                            ->previewable(false),

                        FileDownload::make('download')
                            ->label('')
                            ->proxyRoute('proxy.download')
                            ->disk(config('saas-deadman.attachments_disk'))
                            ->fileNames(fn($get) => $get('attachment'))
                    ])
            ]);
    }
}
