<?php

namespace Netzperfekt\SaasDeadman\Filament\Resources\SwitchResource\RelationManagers;

use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Support\Enums\IconSize;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Netzperfekt\SaasDeadman\Filament\Resources\ActionResource\ActionResource;
use Netzperfekt\SaasDeadman\Models\Channel;
use Netzperfekt\SaasDeadman\Models\Contact;

class ActionsRelationManager extends RelationManager
{
    protected static string $relationship = 'actions';
    protected static ?string $inverseRelationship = 'switches';

    protected $appends = array('fullname');

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return __('deadman-actions');
    }

    public function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema($this->getPivotFormSchema());
    }

    public function getPivotFormSchema(bool $withMainFields = false): array
    {
        $fields = [
            Select::make('channel_id')
                ->label(__('deadman-channel'))
                ->options(Channel::all()->pluck('title', 'id'))
                ->required(),

            Select::make('notification_contacts')
                ->label(__('deadman-contacts'))
                ->options(Contact::onlyMyRecords()->withoutMyself()->orderBy('lastname')->get()->pluck('fullname', 'id'))
                ->multiple()
                ->required()
        ];

        if($withMainFields) {
            array_push($fields,
                TextInput::make('subject')
                    ->label(__('deadman-subject'))
                    ->required()
            );
        }

        return $fields;
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('subject')
            ->description(__('deadman-actions-description'))
            ->allowDuplicates()

            ->columns([
                Tables\Columns\TextColumn::make('type')
                    ->label(__('deadman-type'))
                    ->url(fn($record) => ActionResource::getUrl('edit', [$record->action_id]))
                    ->openUrlInNewTab()
                    ->badge(),

                Tables\Columns\TextColumn::make('channel_id')
                    ->formatStateUsing(fn ($state) => Channel::find($state)->title)
                    ->label(__('deadman-channel'))
                    ->sortable(),

                Tables\Columns\TextColumn::make('subject')
                    ->label(__('deadman-subject'))
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('notification_contacts')
                    ->label(__('deadman-contacts'))
                    ->badge()
                    ->color('warning')
                    ->limitList(2)
                    ->formatStateUsing(function ($state)
                    {
                        return Contact::where('id', (int)$state)->get()
                            ->map(fn($contact) => $contact->fullname)
                            ->first();
                    })
                ])

            ->filters([
                //
            ])

            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->form($this->getPivotFormSchema(true)),

                Tables\Actions\AttachAction::make()
                    ->preloadRecordSelect()
                    ->recordSelect(
                        fn (Select $select) => $select->placeholder('Benachrichtigung auswählen')
                            ->multiple()
                    )
                    ->recordSelectOptionsQuery(fn(Builder $query) => $query->onlyMyRecords())
                    ->form(fn (Tables\Actions\AttachAction $action): array => [
                        $action->getRecordSelect(),
                        ...$this->getPivotFormSchema(false)
                    ])
            ])

            ->actions([
                Tables\Actions\EditAction::make()
                    ->label('')
                    ->tooltip('Bearbeiten')
                    ->iconSize(IconSize::Medium)
                    ->form($this->getPivotFormSchema(false)),

                Tables\Actions\DetachAction::make()
                    ->label('')
                    ->tooltip('Trennen')
                    ->iconSize(IconSize::Medium)
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DetachBulkAction::make()
                ]),
            ]);
    }
}
