<?php

namespace Netzperfekt\SaasDeadman\Filament\Resources\SwitchResource;

use Carbon\Carbon;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Tabs\Tab;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Guava\FilamentKnowledgeBase\Contracts\HasKnowledgeBase;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasDeadman\Enums\ConfirmationFrequencyGrace;
use Netzperfekt\SaasDeadman\Enums\SwitchStateEnum;
use Netzperfekt\SaasDeadman\Facades\SwitchService;
use Netzperfekt\SaasDeadman\Filament\Resources\SwitchResource\RelationManagers\ActionsRelationManager;
use Netzperfekt\SaasDeadman\Enums\ConfirmationFrequency;
use Netzperfekt\SaasDeadman\Enums\SwitchType;
use Netzperfekt\SaasDeadman\Models\Channel;
use Netzperfekt\SaasDeadman\Models\Contact;
use Netzperfekt\SaasDeadman\Models\DMSwitch;
use Netzperfekt\SaasDeadman\Filament\Resources\SwitchResource\Pages\CreateSwitch;
use Netzperfekt\SaasDeadman\Filament\Resources\SwitchResource\Pages\EditSwitch;
use Netzperfekt\SaasDeadman\Filament\Resources\SwitchResource\Pages\ListSwitches;
use Netzperfekt\SaasDeadman\States\SwitchGracePeriod;
use Netzperfekt\SaasDeadman\States\SwitchMustConfirm;

class SwitchResource extends Resource implements HasKnowledgeBase
{
    protected static ?string $model = DMSwitch::class;

    protected static ?string $navigationIcon = 'fas-toggle-off';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = -1;
    protected static ?string $tenantRelationshipName = 'switches';

    public static function getNavigationGroup(): ?string
    {
        return __('deadman-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return __('deadman-switch');
    }

    public static function getPluralModelLabel(): string
    {
        return __('deadman-switches');
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Typ' => $record->type->getLabel()
        ];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Split::make([
                    Section::make()

                        // TODO hier ggf. eine andere komponente / button verwenden, das stateIcon mit ausgeben
                        ->description(function($get)
                        {
                            if($get('state') != SwitchStateEnum::Idle->value)
                            {
                                $switchState = SwitchStateEnum::from($get('state'));
                                $stateLabel = $switchState->getLabel();
                                $stateBackground = $switchState->getBackgroundColor();

                                return new HtmlString(
                                    '<div class="' . $stateBackground . ' p-2 font-semibold">' . $stateLabel . '</div>'
                                );
                            }

                            return '';
                        })
                        ->schema([
                            Grid::make()
                                ->columns(4)
                                ->schema([
                                    TextInput::make('title')
                                        ->label(__('deadman-title'))
                                        ->extraAttributes(['class' => 'font-semibold'])
                                        ->autofocus()
                                        ->required()
                                        ->maxLength(255),

                                    Textarea::make('description')
                                        ->label(__('deadman-description'))
                                        ->columnSpan(3)
                                        ->rows(3)
                                ]),

                            Tabs::make()->tabs([
                                Tab::make(__('deadman-confirmation'))
                                    ->schema([
                                        Select::make('channels')
                                            ->label(__('deadman-confirmation-channels'))
                                            ->helperText(__('deadman-confirmation-channels-hint'))
                                            ->live()
                                            ->required()
                                            ->preload()
                                            ->multiple()
                                            ->relationship(
                                                titleAttribute: 'title'
                                            )
                                            ->getOptionLabelFromRecordUsing(function (Channel $record)
                                            {
                                                $contactMyself = Contact::onlyMyRecords()->ownContact()->first();

                                                $channelRecipients = $contactMyself->getAllRecipientsForChannel($record);

                                                $recipientsString = $record->title;
                                                if(count($channelRecipients) > 0)
                                                {
                                                     $recipientsString .= ' (' . implode(', ', $channelRecipients) . ')' ;
                                                }
                                                else
                                                {
                                                    $recipientsString .= ' (---)';
                                                }
                                                return new HtmlString(
                                                    (!$record->active ? '<s title="inaktiv">' : '') .
                                                    $recipientsString .
                                                    (!$record->active ? '</s>' : '')
                                                );
                                            })
                                            ->allowHtml()
                                            ->suffixIcon('far-message'),

                                        Textarea::make('confirmation_message_owner')
                                            ->label(__('deadman-message-owner'))
                                            ->rows(3)
                                            ->columnSpanFull(),
                                    ]),

                                Tab::make(__('deadman-confirmers'))
                                    ->schema([
                                        Grid::make()
                                            ->columns(3)
                                            ->schema([
                                                Select::make('confirmers')
                                                    ->label(__('deadman-confirmers'))
                                                    ->live()
                                                    ->preload()
                                                    ->multiple()
                                                    ->columnSpan(2)
                                                    ->relationship(
                                                        titleAttribute: 'lastname',
                                                        modifyQueryUsing: fn (Builder $query) => $query->onlyMyRecords()->withoutMyself()
                                                    )
                                                    ->getOptionLabelFromRecordUsing(fn (Contact $record) => "{$record->lastname}, {$record->firstname}")
                                                    ->suffixIcon('fas-person-circle-check'),

                                                Select::make('confirmation_mode')
                                                    ->label(__('deadman-confirmation-mod'))
                                                    ->hintIcon('fas-circle-info')
                                                    ->hintIconTooltip(__('deadman-confirmation-mode-hint'))
                                                    ->options(function($get)
                                                    {
                                                        $options = [];
                                                        $maxConfirmersCount = count($get('confirmers'));
                                                        for($i = 1; $i <= $maxConfirmersCount; $i++)
                                                        {
                                                            $options[$i] = __('deadman-enum-confirmation-mode-minimum', ['n' => $i]);
                                                        }
                                                        $options[$maxConfirmersCount + 1] = __('deadman-enum-confirmation-mode-all');

                                                        return $options;
                                                    })
                                                    ->required()
                                                    ->selectablePlaceholder(false)
                                                    ->visible(fn($get) => count($get('confirmers')) > 0),

                                                Toggle::make('confirmation_owner_can_override')
                                                    ->label(__('deadman-confirmation-owner-can_override'))
                                                    ->columnSpan(2)
                                                    ->live()
                                                    ->default(1),

                                                Textarea::make('confirmation_message_additional')
                                                    ->label(__('deadman-message-additional'))
                                                    ->rows(3)
                                                    ->columnSpanFull()
                                            ])
                                    ])
                                ])
                        ])->grow(),

                    Section::make([
                        Toggle::make('active')
                            ->label(__('deadman-active'))
                            ->default(true)
                            ->onColor(Color::Green)
                            ->offColor(Color::Red),

                        Select::make('type')
                            ->label(__('deadman-type'))
                            ->options(SwitchType::class)
                            ->default(SwitchType::Default)
                            ->required(),

                        Select::make('frequency')
                            ->label(__('deadman-confirmation-frequency'))
                            ->hintIcon('fas-info-circle')
                            ->hintIconTooltip(__('deadman-confirmation-frequency-hint'))
                            ->options(ConfirmationFrequency::getOptions([ConfirmationFrequency::Once->value]))
                            ->default(ConfirmationFrequency::Weekly)
                            ->required()
                            ->live()
                            ->afterStateUpdated(function ($livewire, $get, $set, DMSwitch $record)
                            {
                                $nextConfirmationDate = ConfirmationFrequency::from($get('frequency'))
                                    ->getNextConfirmationDate($record->last_confirmation);

                                $set('next_confirmation', $nextConfirmationDate);
                            }),

                        Placeholder::make('frequency_warning')
                            ->label(__('deadman-attention'))
                            ->content(new HtmlString(__('deadman-confirmation-frequency-warning')))
                            ->hintIcon('fas-triangle-exclamation')
                            ->hintColor(Color::Red)
                            ->extraAttributes(['style' => 'background-color: #ffedd5; padding: 10px'])
                            ->visible(fn($get) => ConfirmationFrequency::from($get('frequency'))->showHighFrequencyWarning()),

                        Select::make('frequency_grace')
                            ->label(__('deadman-confirmation-frequency-grace'))
                            ->hintIcon('fas-info-circle')
                            ->hintIconTooltip(__('deadman-confirmation-frequency-grace-hint'))
                            ->options(ConfirmationFrequencyGrace::class)
                            ->default(ConfirmationFrequencyGrace::None)
                            ->required(),

                        DateTimePicker::make('pause_from')
                            ->label(__('deadman-pause-from'))
                            ->placeholder(__('deadman-confirmation-not-set'))
                            ->native(false)
                            ->displayFormat(FORMAT_DISPLAY_DATETIME)
                            ->seconds(false)
                            ->minDate(now()),

                        DateTimePicker::make('pause_until')
                            ->label(__('deadman-pause-until'))
                            ->placeholder(__('deadman-confirmation-not-set'))
                            ->native(false)
                            ->displayFormat(FORMAT_DISPLAY_DATETIME)
                            ->seconds(false)
                            ->minDate(now()),

                        TextInput::make('last_confirmation')
                            ->label(__('deadman-last-confirmation'))
                            ->placeholder(__('deadman-confirmation-not-set'))
                            ->disabled()
                            ->visible(fn($get) => $get('state') != SwitchStateEnum::Triggered->value)
                            ->formatStateUsing(fn(DMSwitch $record) =>
                                $record->last_confirmation?->formatWithTimezone(FORMAT_DISPLAY_DATETIME) ?? '-'),

                        TextInput::make('next_confirmation')
                            ->label(fn($get) =>
                                $get('state') == SwitchStateEnum::Idle->value ?
                                    __('deadman-next-check') :
                                    __('deadman-next-confirmation'))
                            ->placeholder(__('deadman-confirmation-not-set'))
                            ->readOnly()
                            ->visible(fn($get) =>
                                $get('state') != SwitchStateEnum::Triggered->value)
                            ->formatStateUsing(fn(DMSwitch $record) =>
                                $record->confirmUntil?->formatWithTimezone(FORMAT_DISPLAY_DATETIME) ?? '-'),

                        Select::make('state')
                            ->label(__('deadman-state'))
                            ->options(SwitchStateEnum::class)
                            ->disabled()
                            ->default(SwitchStateEnum::Idle->value)
                            ->selectablePlaceholder(false)
                            ->prefixIcon(fn($state) => SwitchStateEnum::from($state)->getIcon())
                            ->extraAttributes(['style' => 'background-color: #ffedd5']),

                        DateTimePicker::make('triggered_at')
                            ->label(__('deadman-triggered-at'))
                            ->displayFormat(FORMAT_DISPLAY_DATETIME)
                            ->seconds(false)
                            ->disabled()
                            ->visible(fn($get) => $get('state') == SwitchStateEnum::Triggered->value),

                        Placeholder::make('confirmations')
                            ->label(__('deadman-confirmations'))
                            ->visible(fn($get) => count($get('confirmations') ?? []) > 0)
                            ->content(function($get)
                            {
                                $s = '';
                                foreach($get('confirmations') as $contactId => $date)
                                {
                                    if($s != '')
                                    {
                                        $s .= '<br>';
                                    }

                                    $dateFormatted = Carbon::make($date)->formatWithTimezone('d.m.Y H:i');
                                    $contactId = (int)ltrim($contactId, '_');
                                    if($contactId == 0)
                                    {
                                        $s .= '<i>Eigentümer</i>: ' . $dateFormatted;
                                    }
                                    else
                                    {
                                        $contact = Contact::find($contactId);
                                        $s .= '<i>' . ($contact?->fullName ?? '---') . '</i>';
                                        $s .= ': ' . $dateFormatted;
                                    }
                                }

                                return new HtmlString($s);
                            })

                    ])->grow(false)
                ])->from('xl')
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('deadman-title'))
                    ->weight(FontWeight::SemiBold)
                    ->tooltip(fn($record) => $record->description)
                    ->searchable(),

                Tables\Columns\TextColumn::make('type')
                    ->label(__('deadman-type'))
                    ->badge(),

                Tables\Columns\TextColumn::make('frequency')
                    ->label(__('deadman-confirmation-frequency'))
                    ->badge()
                    ->color(Color::Blue),

                Tables\Columns\TextColumn::make('next_confirmation')
                    ->label(__('deadman-next-confirmation'))
                    ->placeholder(__('deadman-confirmation-not-set'))
                    ->icon(function(DMSwitch $record)
                    {
                        if($record->next_confirmation < Carbon::now())
                        {
                            return 'heroicon-s-exclamation-circle';
                        }

                        return 'heroicon-o-check-circle';
                    })
                    ->iconColor(function(DMSwitch $record)
                    {
                        if($record->next_confirmation < Carbon::now())
                        {
                            return Color::Red;
                        }

                        return Color::Green;
                    })
                    ->formatStateUsing(fn(DMSwitch $record) =>
                        $record->confirmUntil?->formatWithTimezone(FORMAT_DISPLAY_DATETIME) ?? '-'),

                Tables\Columns\TextColumn::make('state')
                    ->label(__('deadman-state'))
                    ->badge()
                    ->formatStateUsing(fn ($record) => SwitchStateEnum::from($record->state))
                    ->tooltip(function ($record, $state)
                    {
                        if($state == SwitchStateEnum::MustConfirm)
                        {
                            return __('deadman-confirm-click');
                        }
                        elseif($state == SwitchStateEnum::Triggered) {
                            return __('deadman-triggered-at-date', ['date' => $record?->triggered_at?->formatWithTimezone(FORMAT_DISPLAY_DATETIME) ?? '---']);
                        }

                        return '';
                    })
                    ->action(
                        Tables\Actions\Action::make('confirm-now')
                            ->label(__('deadman-confirm-now'))
                            ->iconButton()
                            ->tooltip(__('deadman-confirm-now'))
                            ->icon('heroicon-o-check')
                            ->requiresConfirmation()
                            ->visible(fn(DMSwitch $switch) => $switch->state->equals(SwitchMustConfirm::class) || $switch->state->equals(SwitchGracePeriod::class))
                            ->action(function (DMSwitch $switch)
                            {
                                SwitchService::setConfirmed($switch, 'backend');

                                Notification::make()
                                    ->title(__('deadman-switch-confirmed'))
                                    ->success()
                                    ->send();
                            })
                    ),

                Tables\Columns\TextColumn::make('pause')
                    ->label(__('deadman-paused'))
                    ->getStateUsing(fn(DMSwitch $record) =>
                        $record->pause_from . ' ' . $record->pause_until
                    )
                    ->formatStateUsing(fn(DMSwitch $record) =>
                        ($record->pause_from || $record->pause_until) ?
                            $record->pause_from?->formatWithTimezone(FORMAT_DISPLAY_DATETIME) . ' - ' .
                            $record->pause_until?->formatWithTimezone(FORMAT_DISPLAY_DATETIME)
                            : ''
                    ),

                Tables\Columns\TextColumn::make('actions_count')
                    ->counts('actions')
                    ->label(__('deadman-actions-count'))
                    ->tooltip(fn($record) =>
                        implode(', ',
                            $record->load('actions')
                                ->actions->select('subject')
                                ->pluck('subject')->toArray())
                    )
                    ->badge()
                    ->color(Color::Blue),

                Tables\Columns\ToggleColumn::make('active')
                    ->label(__('deadman-active'))
                    ->onColor(Color::Green)
                    ->offColor(Color::Red)
                    ->disabled()
            ])

            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('type')
                    ->label(__('deadman-type'))
                    ->options(SwitchType::class)
            ])

            ->recordClasses(fn(DMSwitch $switch) => SwitchStateEnum::from($switch->state)->getBackgroundColor())

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),
                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])

            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ])

            ->defaultSort('title');
    }

    public static function getRelations(): array
    {
        return [
            ActionsRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListSwitches::route('/'),
            'create' => CreateSwitch::route('/create'),
            'edit' => EditSwitch::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    public static function getDocumentation(): array
    {
        return [
            'switches'
        ];
    }
}
