<?php

namespace Netzperfekt\SaasDeadman\Models;

use Database\Factories\ContactFactory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasBase\Traits\HasDefaultScopes;
use Netzperfekt\SaasBase\Traits\HasGlobalModelObserver;
use Netzperfekt\SaasDeadman\Enums\ConfirmationFrequency;
use Netzperfekt\SaasDeadman\Enums\ContactType;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Contact extends Model
{
    use HasFactory;
    use SoftDeletes;
    use HasGlobalModelObserver;
    use HasDefaultScopes;

    protected $table = 'deadman_contacts';

    protected $fillable = [
        'id', 'deleted_at', 'created_at', 'updated_at', 'team_id',
        'type',
        'lastname',
        'firstname',
        'company',
        'street',
        'zip',
        'city',
        'country',
        'comments',
        'must_confirm',
        'confirmation_frequency',
        'last_confirmation',
        'next_confirmation'
    ];

    protected $casts = [
        'type' => ContactType::class,
        'confirmation_frequency' => ConfirmationFrequency::class,
        'last_confirmation' => 'datetime',
        'next_confirmation' => 'datetime',
    ];

    public function getFullnameAttribute()
    {
        return $this->lastname . ', ' . $this->firstname;
    }

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function recipients(?Channel $channel = null): HasMany
    {
        $q = $this->hasMany(ContactRecipient::class, 'contact_id');
        if($channel)
        {
            $q->where('channel_id', $channel->id);
        }

        return $q;
    }

    public function getAllRecipientsForChannel(Channel $channel): array
    {
        $allRecipients = [];
        foreach($this->recipients($channel)->get() as $recipient)
        {
            $allRecipients = array_merge($allRecipients, $recipient->recipients);
        }

        return $allRecipients;
    }

    public function switches(): BelongsToMany
    {
        return $this->belongsToMany(
            DMSwitch::class,
            'deadman_contact_switches',
            'contact_id',
            'switch_id',
        )
            ->withPivot(['id', 'switch_config', 'sort_order'])
            ->withCasts(['switch_config' => 'array'])
            ->using(ContactSwitches::class)
            ->orderBy('sort_order');
    }

    public function switchConfirmers(): BelongsToMany
    {
        return $this->belongsToMany(
            DMSwitch::class,
            'deadman_switch_confirmers',
            'contact_id',
            'switch_id',
        )
            ->withPivot(['id', 'confirmer_config', 'sort_order'])
            ->withCasts(['confirmer_config' => 'array'])
            ->using(SwitchChannels::class)
            ->orderBy('sort_order');
    }

    public function notificationTargets(): BelongsToMany
    {
        return $this->belongsToMany(
            Channel::class,
            'deadman_contact_notification_targets',
            'contact_id',
            'channel_id',
        )
            ->withPivot(['id', 'active', 'subject', 'template', 'sort_order'])
            ->using(ContactNotificationTargets::class)
            ->orderBy('sort_order');
    }

    public function scopeOwnContact(Builder $query, ?Team $team = null): void
    {
        $query->whereIn('type', [ ContactType::Myself ]);

        if($team)
        {
            $query->where('team_id', '=', $team->id);
        }
    }

    public function scopeWithoutMyself(Builder $query): void
    {
        $query->whereNotIn('type', [ ContactType::Myself ]);
    }

    public function isOwnContact(): bool
    {
        return $this->type == ContactType::Myself;
    }

    protected static function newFactory(): Factory
    {
        return ContactFactory::new();
    }
}
