<?php

namespace Netzperfekt\SaasDeadman\Models;

use Carbon\Carbon;
use Database\Factories\DMSwitchFactory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasBase\Traits\HasDefaultScopes;
use Netzperfekt\SaasBase\Traits\HasGlobalModelObserver;
use Netzperfekt\SaasDeadman\Enums\ConfirmationFrequency;
use Netzperfekt\SaasDeadman\Enums\ConfirmationFrequencyGrace;
use Netzperfekt\SaasDeadman\Enums\SwitchType;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Netzperfekt\SaasDeadman\States\SwitchGracePeriod;
use Netzperfekt\SaasDeadman\States\SwitchIdle;
use Netzperfekt\SaasDeadman\States\SwitchMustConfirm;
use Netzperfekt\SaasDeadman\States\SwitchState;
use Netzperfekt\SaasDeadman\States\SwitchTriggered;
use Spatie\ModelStates\HasStates;

class DMSwitch extends Model
{
    use HasFactory;
    use SoftDeletes;
    use HasGlobalModelObserver;
    use HasDefaultScopes;
    use HasStates;

    protected $table = 'deadman_switches';

    protected $fillable = [
        'id', 'deleted_at', 'created_at', 'updated_at', 'team_id',
        'type',
        'title',
        'description',
        'active',
        'frequency',
        'frequency_grace',
        'confirmation_mode',
        'confirmation_owner_can_override',
        'confirmations',
        'confirmation_message_owner',
        'confirmation_message_additional',
        'pause_from',
        'pause_until',
        'last_confirmation',
        'next_confirmation',
        'next_confirmation_with_grace',
        'state',
        'triggered_at'
    ];

    protected $casts = [
        'type' => SwitchType::class,
        'frequency' => ConfirmationFrequency::class,
        'frequency_grace' => ConfirmationFrequencyGrace::class,
        'state' => SwitchState::class,
        'pause_from' => 'datetime',
        'pause_until' => 'datetime',
        'last_confirmation' => 'datetime',
        'next_confirmation' => 'datetime',
        'next_confirmation_with_grace' => 'datetime',
        'triggered_at' => 'datetime',
        'confirmations' => 'array'
    ];

    public function getConfirmUntilAttribute()
    {
        if($this->state->equals(SwitchGracePeriod::class))
        {
            return $this->next_confirmation_with_grace;
        }

        return $this->next_confirmation;
    }


    public function scopeActive(Builder $query): void
    {
        $query->where('active', 1);
    }

    public function scopeNotPaused(Builder $query): void
    {
        $now = Carbon::now();

        $query
            ->whereNot(function($notQuery) use ($now)
            {
                $notQuery->whereDate('pause_from', '<=', $now)
                         ->whereDate('pause_until', '>=', $now);
            })
            ->orWhere(function($orQuery)
            {
                $orQuery->whereNull('pause_from')
                        ->whereNull('pause_until');
            });
    }

    public function scopeOverdue(Builder $query): void
    {
        $query->whereDate(
            'next_confirmation',
            '<',
            Carbon::now()
        );
    }

    public function scopeIdle(Builder $query): void
    {
        $query->whereState('state', SwitchIdle::class)
              ->orWhereNull('state');
    }

    public function scopeInGracePeriod(Builder $query): void
    {
        $query->whereState('state', SwitchGracePeriod::class)
            ->orWhereNull('state');
    }

    public function scopeMustConfirm(Builder $query): void
    {
        $query->whereState('state', SwitchMustConfirm::class);
    }

    public function scopeMustConfirmOrGrace(Builder $query): void
    {
        $query->whereState('state', [SwitchMustConfirm::class, SwitchGracePeriod::class]);
    }

    public function scopeTriggered(Builder $query): void
    {
        $query->whereState('state', SwitchTriggered::class);
    }

    public function scopeNotTriggered(Builder $query): void
    {
        $query->whereNotState('state', SwitchTriggered::class);
    }

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function contacts(): BelongsToMany
    {
        return $this->belongsToMany(
            Contact::class,
            'deadman_contact_switches',
            'switch_id',
            'contact_id',
        )
            ->withPivot(['id', 'switch_config', 'sort_order'])
            ->withCasts(['switch_config' => 'array'])
            ->using(ContactSwitches::class)
            ->orderBy('sort_order');
    }

    public function channels(): BelongsToMany
    {
        return $this->belongsToMany(
            Channel::class,
            'deadman_switch_channels',
            'switch_id',
            'channel_id',
        )
            ->withPivot(['id', 'channel_config', 'confirmation_sent', 'sort_order'])
            ->withCasts(['channel_config' => 'array', 'confirmation_sent' => 'datetime'])
            ->using(SwitchChannels::class)
            ->orderBy('sort_order');
    }

    public function actions(): BelongsToMany
    {
        return $this->belongsToMany(
            Action::class,
            'deadman_switch_actions',
            'switch_id',
            'action_id',
        )
            ->withPivot(['id', 'channel_id', 'notification_contacts', 'notification_sent'])
            ->withCasts(['notification_contacts' => 'array'])
            ->using(SwitchActions::class);
    }

    public function confirmers(): BelongsToMany
    {
        return $this->belongsToMany(
            Contact::class,
            'deadman_switch_confirmers',
            'switch_id',
            'contact_id',
        )
            ->withPivot(['id', 'confirmer_config', 'sort_order'])
            ->withCasts(['confirmer_config' => 'array'])
            ->using(SwitchConfirmers::class)
            ->orderBy('sort_order');
    }

    public function updateNextConfirmationDates(bool $setLastConfirmationToNow = false): void
    {
        if($setLastConfirmationToNow)
        {
            $this->last_confirmation = Carbon::now();
        }

        $this->next_confirmation = $this->frequency->getNextConfirmationDate($this->last_confirmation);

        if($this->hasGracePeriod())
        {
            $this->next_confirmation_with_grace = $this->frequency_grace->addGracePeriod(
                Carbon::now()
            );
        }
    }

    public function isActive(): bool
    {
        return $this->active;
    }

    public function isOverdue(bool $respectGracePeriod = true): bool
    {
        if($this->hasGracePeriod() && $respectGracePeriod)
        {
            return $this->next_confirmation_with_grace == null ||
                   $this->next_confirmation_with_grace->lessThan(Carbon::now());
        }

        return $this->next_confirmation == null ||
               $this->next_confirmation->lessThan(Carbon::now());
    }

    public function isIdle(): bool
    {
        return $this->state->equals(SwitchIdle::class);
    }

    public function isInGracePeriod(): bool
    {
        return $this->state->equals(SwitchGracePeriod::class);
    }

    public function isMustConfirm(): bool
    {
        return $this->state->equals(SwitchMustConfirm::class);
    }

    public function isTriggered(): bool
    {
        return $this->state->equals(SwitchTriggered::class);
    }

    public function hasGracePeriod(): bool
    {
        return $this->frequency_grace != ConfirmationFrequencyGrace::None;
    }

    // sets confirmations for a specific contactId
    public function confirmFromContact(int $contactId): void
    {
        $confirmations = $this->confirmations;
        $confirmations['_' . $contactId] = Carbon::now();
        $this->confirmations = $confirmations;

        $this->save();
    }

    // has a specific contactId already confirmed this switch?
    public function isConfirmedForContact(int $contactId): bool
    {
        return $this->confirmations['_' . $contactId] ?? false;
    }

    public function isCompletelyConfirmed()
    {
        $ownerHasConfirmed = array_key_exists('_0', $this->confirmations ?? []);
        $countConfirmed = count($this->confirmations ?? []);

        // owner can confirm by himself and has the already confirmed?
        if($this->confirmation_owner_can_override && $ownerHasConfirmed)
        {
            return true;
        }

        // otherwise: if there are ANY confirmations (owner and/or other confirmers)
        // then the confirmation_mode (= minimum number of confirmations needed) must be equal or exceeded
        return $countConfirmed > 0 && $countConfirmed >= $this->confirmation_mode;
    }

    public function dump(bool $dumpAndDie = false)
    {
        dump('title: ' . $this->title);
        dump('now: ' . Carbon::now()->format('d.m.Y H:i'));
        dump('last conf:' .  $this->last_confirmation?->format('d.m.Y H:i') ?? '-');
        dump('next conf:' .  $this->next_confirmation?->format('d.m.Y H:i') ?? '-');
        dump('next conf grace: ' . $this->next_confirmation_with_grace?->format('d.m.Y H:i') ?? '-');
        dump('state: ' . $this->state->getValue());
        dump('---');

        if($dumpAndDie)
        {
            die('');
        }
    }

    protected static function newFactory(): Factory
    {
        return DMSwitchFactory::new();
    }
}
