<?php

namespace Netzperfekt\SaasDeadman\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Slack\BlockKit\Blocks\SectionBlock;
use Illuminate\Notifications\Slack\SlackMessage;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasDeadman\Models\Channel;
use Netzperfekt\SaasDeadman\Models\Contact;
use Netzperfekt\SaasDeadman\Models\DMSwitch;

class Confirmation extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public readonly DMSwitch $switch,
                                public readonly Channel $channel,
                                public readonly ?Contact $additionalConfirmer = null,
                                public readonly bool $inGracePeriod = false)
    {
        $this->queue = config('saas-deadman.queue_confirmations');
    }

    public function via(object $notifiable): array
    {
        return [ $this->channel->type->getNotificationChannel() ];
    }

    public function toSlack(object $notifiable): SlackMessage
    {
        $firstUser = $this->switch->team->members->first();
        $confirmationUrl = $this->getConfirmationUrl($this->additionalConfirmer);

        $additionalSubject = $this->inGracePeriod ? __('deadman-mail-confirmation-grace-period') : '';
        $confirmUntilFormatted = $this->switch->confirmuntil->formatWithTimezone(FORMAT_DISPLAY_DATETIME, $firstUser->timezone);

        if($this->additionalConfirmer)
        {
            $msg = (new SlackMessage)
                ->headerBlock(
                    __('deadman-mail-confirmation-subject', [
                        'title' => $this->switch->title . ' / ' . $firstUser->name,
                        'additional' => $additionalSubject
                    ]
                ))

                ->sectionBlock(function (SectionBlock $block)
                {
                    $block->text(__('deadman-mail-confirmation-greeting-additional-confirmer', [
                        'name' => $this->additionalConfirmer->fullname
                    ]));
                })

                ->sectionBlock(function (SectionBlock $block) use ($firstUser, $confirmUntilFormatted)
                {
                    if (!empty($this->switch->confirmation_message_additional))
                    {
                        $block->text(__('deadman-mail-confirmation-message-md', [
                            'message' => $this->switch->confirmation_message_additional
                        ]))->markdown();
                    }
                    else
                    {
                        $block->text(
                            __('deadman-mail-confirmation-request-additional-confirmer-md', ['date' => $confirmUntilFormatted])
                        )->markdown();
                    }

                    $block->field(__('deadman-mail-confirmation-switch-owner-md', [
                        'ownerName' => $firstUser->name
                    ]))->markdown();
                });
        }

        // owner
        else
        {
            $msg = (new SlackMessage)
                ->headerBlock(
                    __('deadman-mail-confirmation-subject', [
                        'title' => $this->switch->title . ' / ' . $firstUser->name,
                        'additional' => $additionalSubject
                    ]))

                ->sectionBlock(function (SectionBlock $block) use($firstUser)
                {
                    $block->text(__('deadman-mail-confirmation-greeting', [
                        'name' => $firstUser->name
                    ]));
                })

                ->sectionBlock(function (SectionBlock $block) use ($firstUser, $confirmUntilFormatted)
                {
                    if (!empty($this->switch->confirmation_message_owner))
                    {
                        $block->text(__('deadman-mail-confirmation-message-md', [
                            'message' => $this->switch->confirmation_message_owner
                        ]))->markdown();
                    }
                    else
                    {
                        $block->text(
                            __('deadman-mail-confirmation-request-md', ['date' => $confirmUntilFormatted])
                        )->markdown();
                    }

                    $block->field(__('deadman-mail-confirmation-switch-owner-md', [
                        'ownerName' => $firstUser->name
                    ]))->markdown();
                });

        }

        $msg->sectionBlock(function (SectionBlock $block) use ($confirmationUrl)
        {
            $block->field('<' . $confirmationUrl . '|*' . __('deadman-mail-confirmation-action') . '*>')
                ->markdown();
            $block->text(__('deadman-mail-confirmation-outro'));
        });

        return $msg;
    }

    public function toMail(object $notifiable): MailMessage
    {
        $firstUser = $this->switch->team->members->first();
        $confirmationUrl = $this->getConfirmationUrl($this->additionalConfirmer);

        $msg = (new MailMessage)->priority(1); // highest

        $additionalSubject = $this->inGracePeriod ? __('deadman-mail-confirmation-grace-period') : '';
        $confirmUntilFormatted = $this->switch->confirmuntil->formatWithTimezone(FORMAT_DISPLAY_DATETIME, $firstUser->timezone);

        if($this->additionalConfirmer)
        {
            $msg->subject(__('deadman-mail-confirmation-subject', [
                'title' => $this->switch->title . ' / ' . $firstUser->name,
                'additional' => $additionalSubject
            ]))
                ->greeting(new HtmlString(__('deadman-mail-confirmation-greeting-additional-confirmer', [
                    'name' => $this->additionalConfirmer->fullname
                ])));

            if (!empty($this->switch->confirmation_message_additional))
            {
                $msg->line(new HtmlString(__('deadman-mail-confirmation-message', [
                    'message' => new HtmlString($this->switch->confirmation_message_additional)
                ])));
            }

            $msg->line(new HtmlString(
                __('deadman-mail-confirmation-request-additional-confirmer', ['date' => $confirmUntilFormatted])
            ));

            $msg->Line(new HtmlString(__('deadman-mail-confirmation-switch-owner', [
                'ownerName' => $firstUser->name
            ])));
        }

        // owner
        else
        {
            $msg->subject(__('deadman-mail-confirmation-subject', [
                'title' => $this->switch->title,
                'additional' => $additionalSubject
            ]))
                ->greeting(new HtmlString(__('deadman-mail-confirmation-greeting', ['name' => $firstUser->name])));

            if (!empty($this->switch->confirmation_message_owner))
            {
                $msg->line(new HtmlString(__('deadman-mail-confirmation-message', [
                    'message' => new HtmlString($this->switch->confirmation_message_owner)
                ])));
            }

            $msg->line(new HtmlString(
                __('deadman-mail-confirmation-request', ['date' => $confirmUntilFormatted])
            ));
        }
        
        $msg->action(new HtmlString(__('deadman-mail-confirmation-action')), $confirmationUrl)
            ->line(new HtmlString(__('deadman-mail-confirmation-outro')));
            
        return $msg;
    }

    public function toArray(object $notifiable): array
    {
        return [
            //
        ];
    }

    private function getConfirmationUrl(?Contact $contact)
    {
        return URL::temporarySignedRoute(
            'front.switch.confirm',
            now()->addHours(config('saas-deadman.switch_confirmation_validity')),
            [
                'switchId' => $this->switch->id,
                'contactId' => empty($contact) ? 0 : $contact->id // owner of the switch
            ]
        );
    }
}
