<?php

namespace Netzperfekt\SaasDeadman\Notifications;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Slack\BlockKit\Blocks\SectionBlock;
use Illuminate\Notifications\Slack\SlackMessage;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasBase\Models\Team;
use Netzperfekt\SaasDeadman\Filament\Resources\ContentBlocks;
use Netzperfekt\SaasDeadman\Models\Action;
use Netzperfekt\SaasDeadman\Models\Channel;
use Netzperfekt\SaasDeadman\Models\Contact;
use Netzperfekt\SaasDeadman\Models\DMSwitch;

class Trigger extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public readonly DMSwitch $switch,
                                public readonly Channel $channel,
                                public readonly Action $notification,
                                public readonly Contact $contact)
    {
        $this->queue = config('saas-deadman.queue_trigger');
    }

    public function via(object $notifiable): array
    {
        return [ $this->channel->type->getNotificationChannel() ];
    }

    public function toSlack(object $notifiable): SlackMessage
    {
        $firstUser = $this->switch->team->members->first();
        $triggerUrl = $this->getTriggerUrl();

        $msg = (new SlackMessage)
            ->headerBlock(
                __('deadman-mail-trigger-subject', [
                    'switch' => $this->switch->title, 'action' => $this->notification->subject
                ])
            )

            ->sectionBlock(function (SectionBlock $block) use ($firstUser)
            {
                $block->field(__('deadman-mail-trigger-greeting', [
                    'name' => $this->contact->firstname . ' ' . $this->contact->lastname
                ]))->markdown();

                $block->field(__('deadman-mail-trigger-intro-md', [
                    'switch' => $this->switch->title, 'action' => $this->notification->subject,
                    'date' => Carbon::now()->formatWithTimezone(FORMAT_DISPLAY_DATETIME, $firstUser->timezone)
                ]))->markdown();
            });

        if($this->notification->attach_contents)
        {
            // TODO in Slack nicht die Inhalte anhängen -> slack unterstützt kein HTML, müsste konvertiert werden,
            //      ist aber ohnehin zu viel für eine slack-nachricht
        }

        $msg->sectionBlock(function (SectionBlock $block) use ($triggerUrl)
        {
            $block->field('<' . $triggerUrl . '|*' . __('deadman-mail-trigger-action') . '*>')
                ->markdown();
            $block->text(__('deadman-mail-trigger-outro'));
        });

        return $msg;
    }

    public function toMail(object $notifiable): MailMessage
    {
        $team = Team::find($this->switch->team_id);
        $firstUser = $this->switch->team->members->first();
        $triggerUrl = $this->getTriggerUrl();

        $msg = (new MailMessage)
            ->subject(__('deadman-mail-trigger-subject', [
                'switch' => $this->switch->title, 'action' => $this->notification->subject
            ]))

            ->greeting(new HtmlString(__('deadman-mail-trigger-greeting', [
                'name' => $this->contact->firstname . ' ' . $this->contact->lastname]
            )))

            ->line(new HtmlString(__('deadman-mail-trigger-intro', [
                'switch' => $this->switch->title, 'action' => $this->notification->subject,
                'date' => Carbon::now()->formatWithTimezone(FORMAT_DISPLAY_DATETIME, $firstUser->timezone)
            ])));

        if($this->notification->attach_contents)
        {
            loginAsTeam($team->id);

            $msg->line(new HtmlString('<hr>'));
            foreach($this->notification->contents as $block)
            {
                $contentsHtml = ContentBlocks::getContentBlockContentsAsHtml($block['type'], $block['data'], $team->id);

                $msg->line(new HtmlString($contentsHtml))
                    ->line(new HtmlString('<hr>'));
            }
        }

        $msg->line(new HtmlString('<br>'))
            ->action(new HtmlString(__('deadman-mail-trigger-action')), $triggerUrl)
            ->line(new HtmlString(__('deadman-mail-trigger-outro')));

        return $msg;
    }

    public function toArray(object $notifiable): array
    {
        return [
            //
        ];
    }

    private function getTriggerUrl(): string
    {
        return URL::signedRoute(
            'front.notification.show', [
                'switchId'  => $this->switch->id,
                'actionId'  => $this->notification->id,
                'contactId' => $this->contact->id,
            ]
        );
    }
}
