<?php

namespace Netzperfekt\SaasDeadman\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Slack\BlockKit\Blocks\SectionBlock;
use Illuminate\Notifications\Slack\SlackMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasDeadman\Models\Channel;
use Netzperfekt\SaasDeadman\Models\Contact;

class VerificationNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public readonly Contact $contact,
                                public readonly Channel $channel)
    {
        $this->queue = config('saas-deadman.queue_confirmations');
    }

    public function via(object $notifiable): array
    {
        return [ $this->channel->type->getNotificationChannel() ];
    }

    public function toSlack(object $notifiable): SlackMessage
    {
        $code = random_int(100000, 999999);

        $firstUser = $this->contact->team->members->first();
        $recipient = $this->getRecipient($notifiable);
        $this->resetConfirmations($recipient, $code);
        $verificationUrl = $this->getConfirmationUrl($recipient);

        return (new SlackMessage)
            ->headerBlock(__('deadman-mail-verify-subject'))

            ->sectionBlock(function (SectionBlock $block) use ($firstUser, $recipient) {
                $block->text(__('deadman-mail-verify-greeting', ['name' => $firstUser->name]));
                $block->field(__('deadman-mail-verify-intro', ['recipient' => $recipient]));
            })

            ->sectionBlock(function (SectionBlock $block) use ($verificationUrl, $code) {
                $block->field(__('deadman-mail-verify-code-md', ['code' => $code]))->markdown();
                $block->text('<' . $verificationUrl . '|*' . __('deadman-mail-verify-action') . '*>')->markdown();
            });
    }

    public function toMail(object $notifiable): MailMessage
    {
        $code = random_int(100000, 999999);

        $firstUser = $this->contact->team->members->first();
        $recipient = $this->getRecipient($notifiable);
        $this->resetConfirmations($recipient, $code);
        $verificationUrl = $this->getConfirmationUrl($recipient);

        return (new MailMessage)
            ->subject(__('deadman-mail-verify-subject'))
            ->greeting(new HtmlString(__('deadman-mail-verify-greeting', ['name' => $firstUser->name])))
            ->line(new HtmlString(__('deadman-mail-verify-intro', ['recipient' => $recipient])))
            ->line(new HtmlString(__('deadman-mail-verify-code', ['code' => $code])))
            ->action(new HtmlString(__('deadman-mail-verify-action')), $verificationUrl);
    }

    public function toArray(object $notifiable): array
    {
        return [
            //
        ];
    }

    private function getRecipient(object $notifiable): string
    {
        $recipient = array_values($notifiable->routes);
        $recipient = array_shift($recipient);

        return $recipient;
    }

    private function getConfirmationUrl(string $recipient): string
    {
        return URL::signedRoute(
            'front.mail.verify', [
                'channelId' => $this->channel->id,
                'contactId' => $this->contact->id,
                'recipient' => $recipient
            ]
        );
    }
    private function resetConfirmations(string $recipient, string $code): void
    {
        foreach($this->contact->recipients($this->channel)->get() as $thisRecipient)
        {
            if(in_array($recipient, $thisRecipient->recipients))
            {
                $confirmations = $thisRecipient->confirmations ?? [];
                $confirmations[$recipient] = [
                    'code'      => $code,
                    'confirmed' => false,
                    'date'      => ''
                ];

                $thisRecipient->confirmations = $confirmations;
                $thisRecipient->save();
            }
        }
    }
}
