<?php

namespace Netzperfekt\SaasDeadman;

use Filament\Support\Assets\Css;
use Filament\Support\Assets\Js;
use Filament\Support\Facades\FilamentAsset;
use Illuminate\Support\Facades\Storage;
use Netzperfekt\SaasDeadman\Commands\CheckSwitchesCommand;
use Netzperfekt\SaasDeadman\Commands\DeleteUnusedAttachmentsCommand;
use Netzperfekt\SaasDeadman\Encryption\EncryptedS3Adapter;
use Netzperfekt\SaasDeadman\Middleware\SetTestingDatabaseConnectionMiddleware;
use Netzperfekt\SaasDeadman\Services\CounterService;
use Netzperfekt\SaasDeadman\Services\SwitchService;
use Spatie\LaravelPackageTools\Commands\InstallCommand;
use Illuminate\Console\Scheduling\Schedule;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;

use Illuminate\Contracts\Foundation\Application;

class SaasDeadmanServiceProvider extends PackageServiceProvider
{
    public function packageRegistered(): void
    {
        $this->app->register(EventServiceProvider::class);
    }

    public function packageBooted(): void
    {
        $this->addFlysystemAdapter();
        $this->addAssets();
        $this->addDocs();
        $this->registerServices();
        $this->registerSchedules();
    }

    public function configurePackage(Package $package): void
    {
        $package
            ->name(SaasDeadman::PACKAGE_NAME)
            ->hasRoute('web')
            ->hasViews()
            ->hasConfigFile()
            ->hasTranslations()
            ->hasCommands([
                DeleteUnusedAttachmentsCommand::class,
                CheckSwitchesCommand::class
            ])

            ->hasMigrations([
                'create_channels_table',
                'create_contacts_table',
                'create_switches_table',
                'create_switch_channels_table',
                'create_switch_confirmers_table',
                'create_contact_switches_table',
                'create_contact_recipients_table',
                'create_contact_notification_targets_table',
                'create_actions_table',
                'create_switch_actions_table'
            ])
            
            ->hasInstallCommand(function(InstallCommand $command)
            {
                $command
                    ->publishAssets()
                    ->publishMigrations()
                    ->publishConfigFile()
                    ->askToRunMigrations();
            });
    }

    private function addAssets()
    {
        FilamentAsset::register([
            Js::make('saas-deadman-js', __DIR__ . '/../resources/dist/saas-deadman.js'),
            Css::make('saas-deadman-css', __DIR__ . '/../resources/dist/saas-deadman.css'),
            Css::make('saas-deadman-frontend-css', __DIR__ . '/../resources/dist/saas-deadman-frontend.css'),
        ], 'netzperfekt/saas-deadman');
    }

    private function addDocs()
    {
        $this->publishes([
            $this->package->basePath('/../docs') => base_path("/docs"),
        ], "{$this->package->shortName()}-docs");

    }

    private function registerServices()
    {
        $this->app->singleton('switch-service', function ()
        {
            return new SwitchService();
        });

        $this->app->singleton('counter-service', function ()
        {
            return new CounterService();
        });
    }

    private function registerSchedules()
    {
        $this->app->booted(function ()
        {
            /** @var Schedule $schedule */
            $schedule = $this->app->make(Schedule::class);
            $schedule->command((new CheckSwitchesCommand)->getSignature())
                ->everyMinute()
                ->withoutOverlapping();
        });
    }
    private function addFlysystemAdapter()
    {
        Storage::extend('s3_encrypted', function(Application $app, array $config)
        {
            return EncryptedS3Adapter::make($app, $config);
        });
    }
}
