<?php

namespace Netzperfekt\SaasDeadman\Services;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Notification as LaravelNotification;
use Netzperfekt\SaasBase\Models\User;
use Netzperfekt\SaasDeadman\Enums\ActivityLogType;
use Netzperfekt\SaasDeadman\Facades\CounterService as CounterServiceAlias;
use Netzperfekt\SaasDeadman\Models\Channel;
use Netzperfekt\SaasDeadman\Models\Contact;
use Netzperfekt\SaasDeadman\Models\DMSwitch;
use Netzperfekt\SaasDeadman\Models\Action;
use Netzperfekt\SaasDeadman\Notifications\Confirmation;
use Netzperfekt\SaasDeadman\Notifications\TestNotification;
use Netzperfekt\SaasDeadman\Notifications\Trigger;
use Netzperfekt\SaasDeadman\Notifications\VerificationNotification;
use Netzperfekt\SaasDeadman\States\SwitchGracePeriod;
use Netzperfekt\SaasDeadman\States\SwitchIdle;
use Netzperfekt\SaasDeadman\States\SwitchMustConfirm;
use Netzperfekt\SaasDeadman\States\SwitchTriggered;

class SwitchService
{
    // sets a switch as confirmed and calculates the next confirmation date
    public function setConfirmed(DMSwitch $switch, string $source = 'web')
    {
        $this->setIdle($switch, $source);

        $switch->updateNextConfirmationDates();
        $switch->last_confirmation = Carbon::now();
        $switch->save();

        activity()
            ->performedOn($switch)
            ->withProperties(['source' => $source])
            ->log(ActivityLogType::SwitchStateConfirmed->value);

        return $switch->next_confirmation;
    }

    public function setIdle(DMSwitch $switch, string $source = 'web')
    {
        $switch->state->transitionTo(SwitchIdle::class);

        activity()
            ->performedOn($switch)
            ->withProperties(['source' => $source])
            ->log(ActivityLogType::SwitchStateIdle->value);
    }

    function checkSwitches(Collection|DMSwitch|null $switchOrSwitches = null): array
    {
        $switches = null;
        if( ! $switchOrSwitches)
        {
            $switches = DMSwitch::active()->notPaused()->notTriggered()->get();
        }
        elseif($switchOrSwitches instanceof DMSwitch)
        {
            $switches = Collection::wrap([$switchOrSwitches]);
        }
        else
        {
            // just to be sure (active + notPaused + notTriggered)
            $switches = $switchOrSwitches->toQuery()->active()->notPaused()->notTriggered()->get();
        }

        CounterServiceAlias::resetCounter();

        foreach($switches as $switch)
        {
            $this->checkSwitchForTransition($switch);
        }

        return CounterServiceAlias::getCounter();
    }

    private function checkSwitchForTransition(DMSwitch $switch)
    {
        $possibleStates = $switch->state->transitionableStates();

        if(in_array(SwitchMustConfirm::$name, $possibleStates))
        {
            $switch->state->transitionTo(SwitchMustConfirm::class);
            $switch->refresh();
            CounterServiceAlias::increaseCounter('mustconfirm');
        }

        elseif(in_array(SwitchGracePeriod::$name, $possibleStates))
        {
            $switch->state->transitionTo(SwitchGracePeriod::class);
            $switch->refresh();
            CounterServiceAlias::increaseCounter('mustconfirm');
        }

        elseif(in_array(SwitchTriggered::$name, $possibleStates))
        {
            $switch->state->transitionTo(SwitchTriggered::class);
            $switch->refresh();
            CounterServiceAlias::increaseCounter('triggered');
        }
    }

    public static function sendTestNotifications(DMSwitch $switch, Channel $channel): int
    {
        $team = $switch->load('team')->team;
        $user = $team->members()->first();
        $contact = Contact::onlyMyRecords()->ownContact($team)->first();

        return self::sendNotification(
            $contact,
            new TestNotification($switch, $channel),
            $switch,
            $channel,
            $user,
            ActivityLogType::TestNotificationSent->value,
            true
        );
    }

    public static function sendVerificationNotifications(Contact $contact, Channel $channel): int
    {
        $user = $contact->load('team')->team->members()->first();

        return self::sendNotification(
            $contact,
            new VerificationNotification($contact, $channel),
            $contact,
            $channel,
            $user,
            ActivityLogType::VerificationNotificationSent->value,
            true
        );
    }

    public static function sendConfirmationNotifications(DMSwitch $switch,
                                                         Channel $channel,
                                                         bool $sendSync = false,
                                                         bool $inGracePeriod = false): int
    {
        $countNotificationsSent = 0;

        $team = $switch->load('team')->team;
        $user = $team->members()->first();

        $switch->confirmations = [];
        $switch->save();

        // --- send confirmation message to switch owner
        $contact = Contact::ownContact($team)->first();

        $countNotificationsSent += self::sendNotification(
            $contact,
            new Confirmation($switch, $channel, inGracePeriod: $inGracePeriod),
            $switch,
            $channel,
            $user,
            ActivityLogType::ConfirmationNotificationsSent->value,
            $sendSync
        );

        // --- send confirmation message to additional confirmers, if there are any
        $additionalConfirmers = $switch->confirmers()->get();
        foreach($additionalConfirmers as $additionalConfirmer)
        {
            $countNotificationsSent += self::sendNotification(
                $additionalConfirmer,
                new Confirmation($switch, $channel, $additionalConfirmer, inGracePeriod: $inGracePeriod),
                $switch,
                $channel,
                $user,
                ActivityLogType::ConfirmationNotificationsSent->value,
                $sendSync
            );
        }

        $channel->pivot->confirmation_sent = Carbon::now();
        $channel->pivot->save();

        return $countNotificationsSent;
    }

    public static function sendTriggerNotifications(DMSwitch $switch, Action $notification): int
    {
        $countNotificationsSent = 0;

        $team = $switch->team;
        $user = $team->members()->first();
        $channel = $notification->pivot->channel;

        // --- send trigger notification to switch owner
        $ownContact = Contact::ownContact($team)->first();

        $countNotificationsSent += self::sendNotification(
            $ownContact,
            new Trigger($switch, $channel, $notification, $ownContact),
            $switch,
            $channel,
            $user,
            ActivityLogType::TriggerNotificationsSent->value
        );

        // --- send trigger notifications to assigned contacts
        $contacts = Contact::whereIn('id', $notification->pivot->notification_contacts)->get();
        foreach($contacts as $contact)
        {
            $countNotificationsSent += self::sendNotification(
                $contact,
                new Trigger($switch, $channel, $notification, $contact),
                $switch,
                $channel,
                $user,
                ActivityLogType::TriggerNotificationsSent->value
            );
        }

        $notification->pivot->notification_sent = Carbon::now();
        $notification->pivot->save();

        return $countNotificationsSent;
    }

    private static function sendNotification(Contact $contact,
                                             Notification $notificationObject,
                                             DMSwitch|Contact $performedOn,
                                             Channel $channel,
                                             User $user,
                                             string $logDescription,
                                             bool $sendSync = false): int
    {
        $recipients = $contact->getAllRecipientsForChannel($channel);
        if(empty($recipients))
        {
            return 0;
        }

        $viaChannel = $channel->type->getNotificationChannel();

        try {
            if($sendSync)
            {
                foreach($recipients as $recipient)
                {
                    LaravelNotification::route($viaChannel, $recipient)->notifyNow($notificationObject);
                }
            }
            else
            {
                foreach($recipients as $recipient)
                {
                    LaravelNotification::route($viaChannel, $recipient)->notify($notificationObject);
                }
            }

            activity()
                ->performedOn($performedOn)
                ->causedBy($user)
                ->withProperties([
                    'channel' => $channel->key,
                    'recipients' => $recipients,
                ])
                ->log($logDescription);
        }
        catch(\Exception $ex)
        {
            activity()
                ->performedOn($performedOn)
                ->causedBy($user)
                ->withProperties([
                    'error' => true,
                    'channel' => $channel->key,
                    'recipients' => $recipients,
                    'message' => $ex->getMessage()
                ])
                ->log($logDescription . ActivityLogType::Error->value);
        }

        return count($recipients);
    }
}
