<?php

use Illuminate\Database\Eloquent\Collection;
use Netzperfekt\SaasDeadman\Enums\ActivityLogType;
use Netzperfekt\SaasDeadman\Facades\SwitchService;
use Netzperfekt\SaasDeadman\Models\Contact;
use Netzperfekt\SaasDeadman\Models\ContactRecipient;
use Netzperfekt\SaasDeadman\States\SwitchMustConfirm;
use function Spatie\PestPluginTestTime\testTime;
use function Pest\Laravel\{get};

describe('switchService', function ()
{
    test('confirm that switch/completelyconfirmed works correctly', function () {
        $recipient1 = 'recipient1@example.com';

        $team = Auth::user()->teams->first();

        $channel = \Netzperfekt\SaasDeadman\Models\Channel::factory()
            ->active()->email()
            ->create();

        $ownContact = Contact::factory()
            ->ownContact()
            ->withTeam($team)
            ->create();

        $confirmer1 = Contact::factory()
            ->withTeam($team)
            ->create();

        $confirmer2 = Contact::factory()
            ->withTeam($team)
            ->create();

        ContactRecipient::factory()
            ->channel($channel)
            ->contact($ownContact)
            ->recipients([$recipient1])
            ->create();

        // only owner, no other confirmers
        $switch1 = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->active()
            ->daily()
            ->withoutGracePeriod()
            ->notPaused()
            ->stateMustConfirm()
            ->needsConfirmation()
            ->confirmationMode(0)
            ->hasAttached($channel)
            ->create();

        // owner + 2 other confirmers - both must confirm, owner can override confirmation
        $switch2 = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->active()
            ->daily()
            ->withoutGracePeriod()
            ->notPaused()
            ->stateMustConfirm()
            ->needsConfirmation()
            ->hasAttached($channel)
            ->confirmationMode(2)
            ->ownerCanOverrideConfirmers()
            ->create();

        \Netzperfekt\SaasDeadman\Models\SwitchConfirmers::make([
            'switch_id' => $switch2->id,
            'contact_id' => $confirmer1->id,
        ])->save();
        \Netzperfekt\SaasDeadman\Models\SwitchConfirmers::make([
            'switch_id' => $switch2->id,
            'contact_id' => $confirmer2->id,
        ])->save();

        // owner + 2 other confirmers - both must confirm, owner can NOT override confirmation
        $switch3 = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->active()
            ->daily()
            ->withoutGracePeriod()
            ->notPaused()
            ->stateMustConfirm()
            ->needsConfirmation()
            ->hasAttached($channel)
            ->confirmationMode(3)
            ->create();

        \Netzperfekt\SaasDeadman\Models\SwitchConfirmers::make([
            'switch_id' => $switch3->id,
            'contact_id' => $confirmer1->id,
        ])->save();
        \Netzperfekt\SaasDeadman\Models\SwitchConfirmers::make([
            'switch_id' => $switch3->id,
            'contact_id' => $confirmer2->id,
        ])->save();

        expect($switch2->confirmers()->get())->toHaveCount(2);
        expect($switch3->confirmers()->get())->toHaveCount(2);

        expect($switch1->isCompletelyConfirmed())->toBeFalse();
        $switch1->confirmFromContact(0); // 0 = owner
        expect($switch1->isCompletelyConfirmed())->toBeTrue();

        $switch2->confirmFromContact(0);
        expect($switch2->isCompletelyConfirmed())->toBeTrue();

        $switch3->confirmFromContact(0);
        expect($switch3->isCompletelyConfirmed())->toBeFalse();

        $switch3->confirmFromContact($confirmer1->id);
        expect($switch3->isCompletelyConfirmed())->toBeFalse();

        $switch3->confirmFromContact($confirmer2->id);
        expect($switch3->isCompletelyConfirmed())->toBeTrue();
    });

    test('confirm that confirmation message is sent to all switches to be confirmed', function ()
    {
        $recipient1 = 'recipient1@example.com';
        $recipient2 = 'recipient2@example.com';

        // ---
        Notification::fake();

        $team = Auth::user()->teams->first();

        $channel = \Netzperfekt\SaasDeadman\Models\Channel::factory()
            ->active()->email()
            ->create();

        $ownContact = Contact::factory()
            ->ownContact()
            ->withTeam($team)
            ->create();

        ContactRecipient::factory()
            ->channel($channel)
            ->contact($ownContact)
            ->recipients([ $recipient1, $recipient2 ])
            ->create();

        $switch1 = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->active()
            ->daily()
            ->withoutGracePeriod()
            ->notPaused()
            ->stateIdle()
            ->needsConfirmation()
            ->hasAttached($channel)
            ->create();

        $switch2 = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->active()
            ->daily()
            ->withoutGracePeriod()
            ->paused()
            ->stateIdle()
            ->needsConfirmation()
            ->hasAttached($channel)
            ->create();

        $switch3 = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->active()
            ->daily()
            ->withGracePeriod4Hours()
            ->notPaused()
            ->stateIdle()
            ->needsConfirmation()
            ->hasAttached($channel)
            ->create();

        $switches = Collection::wrap([
            $switch1, $switch2, $switch3
        ]);
        $counter = SwitchService::checkSwitches($switches);

        $switch1->refresh();
        $switch2->refresh();

        $countMustConfirm = $counter['mustconfirm'];
        $countMustConfirmNotifications = $counter['mustconfirm_notifications'];

        expect($countMustConfirm)->toBe(2);
        expect($countMustConfirmNotifications)->toBe(4);

        $switch1->refresh();
        $switch2->refresh();
        $switch3->refresh();

        expect($switch1->state->equals(SwitchMustConfirm::class))->toBeTrue();
        expect($switch2->state->equals(SwitchMustConfirm::class))->toBeFalse();

        expect($switch1->channels->first()->pivot->confirmation_sent)->not()->toBeNull();

        // test if notifiables where sent
        Notification::assertSentTo(
            new \Illuminate\Notifications\AnonymousNotifiable,
            \Netzperfekt\SaasDeadman\Notifications\Confirmation::class
        );

        Notification::assertCount($countMustConfirmNotifications);

        expect(\Spatie\Activitylog\Models\Activity::all()->last())
            ->toHaveKey('description', ActivityLogType::ConfirmationNotificationsSent->value)
            ->toHaveKey('subject_type', $switch3::class)
            ->toHaveKey('subject_id', $switch3->id);
    });

    test('confirm that sent notification can be retrieved via web', function ()
    {
        // TODO use code from 'confirm that notification is sent to all switches that are triggered'
        //      and check if notification can be retrieved via web
    });

    test('confirm that sent notification contains all contents', function ()
    {
        // TODO use code from 'confirm that notification is sent to all switches that are triggered'
        //      and check if notification message contains all contents
    });

    test('confirm that notification is sent to all switches that are triggered', function ()
    {
        $recipient1 = 'recipient1@example.com';
        $recipient2 = 'recipient2@example.com';
        $recipientOwn = 'recipientOwn@example.com';

        // ---
        Notification::fake();

        $team = Auth::user()->teams->first();

        $channel = \Netzperfekt\SaasDeadman\Models\Channel::factory()
            ->active()
            ->email()
            ->create();

        $ownContact = Contact::factory()
            ->ownContact()
            ->withTeam($team)
            ->create();

        ContactRecipient::factory()
            ->channel($channel)
            ->contact($ownContact)
            ->recipients([ $recipientOwn ])
            ->create();

        $contact1 = \Netzperfekt\SaasDeadman\Models\Contact::factory()
            ->withTeam($team)
            ->create(['type' => \Netzperfekt\SaasDeadman\Enums\ContactType::PrivatePartner]);

        ContactRecipient::factory()
            ->channel($channel)
            ->contact($contact1)
            ->recipients([ $recipient1 ])
            ->create();

        $contact2 = \Netzperfekt\SaasDeadman\Models\Contact::factory()
            ->withTeam($team)
            ->create(['type' => \Netzperfekt\SaasDeadman\Enums\ContactType::Customer]);

        ContactRecipient::factory()
            ->channel($channel)
            ->contact($contact2)
            ->recipients([ $recipient2 ])
            ->create();

        $notification = \Netzperfekt\SaasDeadman\Models\Action::factory()
            ->withTeam($team)
            ->create();

        $switch1 = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->hasAttached($channel)
            ->active()
            ->daily()
            ->withoutGracePeriod()
            ->notPaused()
            ->stateMustConfirm()
            ->overdue()
            ->create();

        // --- muss manuell zusammengebaut werden, sobald 'notification_contacts' insertiert werden soll,
        // werden die spalten nicht korrekt zusammengebaut #wtf
        $switchNotification = \Netzperfekt\SaasDeadman\Models\SwitchActions::make([
            'switch_id' => $switch1->id,
            'action_id' => $notification->id,
            'channel_id' => $channel->id,
        ]);
        $switchNotification->notification_contacts = [$contact1->id, $contact2->id];
        $switchNotification->save();

        $switch2 = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->active()
            ->daily()
            ->withoutGracePeriod()
            ->paused()
            ->stateMustConfirm()
            ->overdue()
            ->create();

        $switches = Collection::wrap([
            $switch1, $switch2
        ]);

        $counter = SwitchService::checkSwitches($switches);

        $switch1->refresh();
        $switch2->refresh();

        $countTriggered = $counter['triggered'];
        $countNotifications = $counter['triggered_notifications'];

        expect($countTriggered)->toBe(1);
        expect($countNotifications)->toBe(3);

        expect($switch1->state->equals(\Netzperfekt\SaasDeadman\States\SwitchTriggered::class))->toBeTrue();
        expect($switch2->state->equals(\Netzperfekt\SaasDeadman\States\SwitchTriggered::class))->toBeFalse();

        expect($switch1->actions->first()->pivot->notification_sent)->not()->toBeNull();

        // test if notifiables where sent
        Notification::assertSentTo(
            new \Illuminate\Notifications\AnonymousNotifiable,
            \Netzperfekt\SaasDeadman\Notifications\Trigger::class
        );
        Notification::assertCount($countNotifications);

        expect(\Spatie\Activitylog\Models\Activity::all()->last())
            ->toHaveKey('description', ActivityLogType::TriggerNotificationsSent->value)
            ->toHaveKey('subject_type', $switch1::class)
            ->toHaveKey('subject_id', $switch1->id);
    });

    it('confirms that a switch can be confirmed via web', function ()
    {
        $recipient = 'recipient1@example.com';

        // ---
        $team = Auth::user()->teams->first();

        $channel = \Netzperfekt\SaasDeadman\Models\Channel::factory()->create();

        $ownContact = Contact::factory()
            ->ownContact()
            ->withTeam($team)
            ->create();

        ContactRecipient::factory()
            ->channel($channel)
            ->contact($ownContact)
            ->recipients([ $recipient ])
            ->create();

        $switch = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->daily()
            ->active()
            ->notPaused()
            ->stateMustConfirm()
            ->overdue()
            ->create();

        $confirmationUrl = \Illuminate\Support\Facades\URL::temporarySignedRoute(
            'front.switch.confirm',
            now()->addHours(config('saas-deadman.switch_confirmation_validity')),
            [
                'switchId' => $switch->id,
                'contactId' => 0
            ]
        );

        $confirmationResponseStep1 = get($confirmationUrl);
        $confirmationUrlStep2 = $confirmationResponseStep1->viewData('url');

        expect($confirmationUrlStep2)->not()->toBeNull();
        get($confirmationUrlStep2)->assertOk();

        $switch->refresh();
        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchIdle::class))->toBeTrue();

        expect($switch)->toHaveKey(
            'next_confirmation',
            $switch->frequency->getNextConfirmationDate($switch->last_confirmation)->format('Y-m-d\TH:i:s.u\Z')
        );

        expect(\Spatie\Activitylog\Models\Activity::all()->last())
            ->toHaveKey('description', ActivityLogType::SwitchConfirmed->value)
            ->toHaveKey('subject_type', $switch::class)
            ->toHaveKey('subject_id', $switch->id);
    });

    test('confirm that switch (without grace period) states are transitioned correctly', function ()
    {
        testTime()->freeze();

        $team = Auth::user()->teams->first();

        $switch = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->active()
            ->notPaused()
            ->weekly()
            ->withoutGracePeriod()
            ->stateIdle()
            ->create();

        $switch->updateNextConfirmationDates();
        $switch->save();
        $switch->refresh();

        // --- check switches -> switch must stay 'idle'
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchIdle::class))->toBeTrue('switch has state "idle"');

        // --- move forward a week -> switch must transition from 'idle' to 'mustconfirm'
        testTime()->addWeek()->subMinute();
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchMustConfirm::class))->toBeFalse('switch has not state "mustConfirm"');

        testTime()->addMinute();
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchMustConfirm::class))->toBeTrue('switch has state "mustConfirm"');

        // --- double check that the switch still stays on "mustConfirm" 5 minutes after
        testTime()->addMinute(5);
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchMustConfirm::class))->toBeTrue('switch still has state "mustConfirm"');
        testTime()->subMinute(5);

        // --- move forward a week -> switch must transition from 'mustconfirm' to 'triggered'
        testTime()->addWeek();
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchTriggered::class))->toBeTrue('switch has state "triggered"');
    });

    test('confirm that switch (with grace period) states are transitioned correctly', function ()
    {
        testTime()->freeze();

        $team = Auth::user()->teams->first();

        $switch = \Netzperfekt\SaasDeadman\Models\DMSwitch::factory()
            ->withTeam($team)
            ->active()
            ->notPaused()
            ->daily()
            ->withGracePeriod4Hours()
            ->stateIdle()
            ->create();

        $switch->updateNextConfirmationDates();
        $switch->save();
        $switch->refresh();

        // --- check switches -> switch must stay 'idle'
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchIdle::class))->toBeTrue('switch has state "idle"');

        // --- move forward a day -> switch must transition from 'idle' to 'mustconfirm'
        testTime()->addDay();
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchMustConfirm::class))->toBeTrue('switch has state "mustConfirm"');

        // --- move forward 4 hours -> switch must transition from 'mustconfirm' to 'graceperiod'
        testTime()->addHours(4)->subMinute();
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchGracePeriod::class))->toBeFalse('switch has not state "grace period"');

        testTime()->addMinute();
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchGracePeriod::class))->toBeTrue('switch has state "grace period"');

        // --- move forward 4 hours -> switch must transition from 'graceperiod' to 'triggered'
        testTime()->addHours(4)->subMinute();
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchTriggered::class))->toBeFalse('switch has not state "triggered"');

        testTime()->addMinute();
        SwitchService::checkSwitches($switch);
        $switch->refresh();

        expect($switch->state->equals(\Netzperfekt\SaasDeadman\States\SwitchTriggered::class))->toBeTrue('switch has state "triggered"');
    });

    // TODO frontend notifications controller test

})->group('feature');
