<?php

namespace Netzperfekt\SaasFaq\Filament\Resources\FaqGroupResource;

use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\ActionSize;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;
use Netzperfekt\SaasFaq\Filament\Resources\FaqGroupResource\RelationManagers\FaqRelationManager;
use Netzperfekt\SaasFaq\Filament\Resources\FaqGroupResource\Pages\CreateFaqGroup;
use Netzperfekt\SaasFaq\Filament\Resources\FaqGroupResource\Pages\EditFaqGroup;
use Netzperfekt\SaasFaq\Filament\Resources\FaqGroupResource\Pages\ListFaqGroup;
use Netzperfekt\SaasFaq\Models\FaqGroup;
use Netzperfekt\SaasFaq\SaasFaq;
use RalphJSmit\Filament\MediaLibrary\Forms\Components\MediaPicker;

class FaqGroupResource extends Resource
{
    protected static ?string $model = FaqGroup::class;

    protected static ?string $navigationIcon = 'heroicon-o-list-bullet';
    protected static ?int $navigationSort = 61;

    public final const TEMPLATE_TYPE_FAQ = 'faq';

    public static function getNavigationGroup(): ?string
    {
        return __('faq-navigation-group');
    }

    public static function getModelLabel(): string
    {
        return __('faq-group');
    }

    public static function getPluralModelLabel(): string
    {
        return __('faq-groups');
    }

    public static function form(Form $form): Form
    {
        $faqModuleDefinition = getFaqModules();
        $thisPackage = SaasFaq::get();

        return $form
            ->columns(3)
            ->schema([
                Section::make('')
                    ->columnSpan(2)
                    ->schema([
                        TextInput::make('title')
                            ->label(__('faq-title'))
                            ->required()
                            ->maxLength(255)
                            ->autofocus()
                            ->live(debounce: 1000)
                            ->afterStateUpdated(function ($get, $set, ?string $state)
                            {
                                if(empty($get('slug'))) {
                                    $set('slug', Str::slug($state));
                                }

                                if(empty($get('seo_title'))) {
                                    $set('seo_title', $state);
                                }
                            }),

                        RichEditor::make('description')
                            ->defaultRichEditor()
                            ->label(__('faq-description'))
                            ->required()
                            ->columns(3),
                    ]),

                Section::make('')
                    ->columnSpan(1)
                    ->schema([
                        TextInput::make('slug')
                            ->label(__('faq-slug'))
                            ->required()
                            ->maxLength(255)
                            ->columns(1),

                        Select::make('module')
                            ->label(__('faq-module'))

                            ->required()
                            ->selectablePlaceholder(fn() => ! $thisPackage->getForTenants())
                            ->default(function() use ($faqModuleDefinition, $thisPackage)
                            {
                                if($thisPackage->getForTenants())
                                {
                                    return $faqModuleDefinition->tenantModules[$thisPackage->getForTenantsPackage()];
                                }

                                return null;
                            })
                            ->options(function() use ($faqModuleDefinition, $thisPackage)
                            {
                                if($thisPackage->getForTenants())
                                {
                                    $tenantModule = $faqModuleDefinition->tenantModules[$thisPackage->getForTenantsPackage()];
                                    return [
                                        $tenantModule => __('faq-module-' . $tenantModule)
                                    ];
                                }
                                else
                                {
                                    return $faqModuleDefinition->modules;
                                }
                            })
                            ->columns(1),

                        Select::make('template')
                            ->label(__('faq-template'))
                            ->relationship(
                                'template',
                                'title',
                                modifyQueryUsing: fn (Builder $query) => $query->where('template_type', self::TEMPLATE_TYPE_FAQ)
                            ),

                        MediaPicker::make('image_id')
                            ->label(__('faq-image'))
                            ->columns(1)
                            ->maxWidth(MaxWidth::ExtraSmall),
                    ]),

                TextInput::make('seo_title')
                    ->label(__('cms-seotitle'))
                    ->columnSpanFull()
                    ->required()
                    ->maxLength(70),

                Textarea::make('seo_description')
                    ->label(__('cms-seodescription'))
                    ->rows(3)
                    ->minLength(120)
                    ->maxLength(300)
                    ->columnSpan(3)
                    ->live(debounce: 250)
                    ->hint(fn ($state, $component) => strlen($state ?? '') . ' / ' . $component->getMaxLength()),

            ]);
    }

    public static function table(Table $table): Table
    {
        $faqModuleDefinition = getFaqModules();

        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('faq-title'))
                    ->weight('font-semibold')
                    ->searchable(),

                Tables\Columns\TextColumn::make('faq_count')
                    ->counts('faq')
                    ->label(__('faq-count'))
                    ->badge(),

                Tables\Columns\TextColumn::make('module')
                    ->label(__('faq-module'))
                    ->badge()
                    ->color('warning')
                    ->formatStateUsing(fn (string $state): string => $faqModuleDefinition->modules[$state] ?? '(' . $state . ')'),

                Tables\Columns\TextColumn::make('template.title')
                    ->label(__('faq-template'))
                    ->color(Color::Orange)
                    ->badge(),


            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('group')
                    ->label(__('faq-group'))
                    ->options($faqModuleDefinition->modules)
            ])
            ->actions([
                Tables\Actions\EditAction::make()->label('')->tooltip('Bearbeiten')->iconSize(IconSize::Medium),

                Tables\Actions\Action::make('preview')
                    ->iconButton()
                    ->tooltip(__('faq-preview'))
                    ->icon('fas-bolt')
                    ->size(ActionSize::Medium)
                    ->color('info')
                    ->url(fn($record) => route('faq.group', $record->slug))
                    ->openUrlInNewTab()
                    ->visible(fn($record) => $record->slug),

                Tables\Actions\DeleteAction::make()->label('')->tooltip('Löschen')->iconSize(IconSize::Medium),
                Tables\Actions\ForceDeleteAction::make()->label('')->tooltip('Endgültig löschen')->iconSize(IconSize::Medium),
                Tables\Actions\RestoreAction::make()->label('')->tooltip('Wiederherstellen')->iconSize(IconSize::Medium)
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            FaqRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListFaqGroup::route('/'),
            'create' => CreateFaqGroup::route('/create'),
            'edit' => EditFaqGroup::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
