<?php

namespace Netzperfekt\SaasFaq\Filament\Resources\FaqResource;

use Filament\Forms\Components\Builder;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;
use Netzperfekt\SaasFaq\Filament\Exporter\ExportAction;
use Netzperfekt\SaasFaq\Filament\Exporter\FaqExporter;
use Netzperfekt\SaasFaq\Filament\Exporter\FaqImporter;
use Netzperfekt\SaasFaq\Filament\Exporter\ImportAction;
use Netzperfekt\SaasFaq\Filament\Resources\FaqResource\Pages\ListFaqPages;
use Netzperfekt\SaasFaq\Filament\Resources\FaqResource\Pages\CreateFaqPage;
use Netzperfekt\SaasFaq\Filament\Resources\FaqResource\Pages\EditFaqPage;
use Netzperfekt\SaasFaq\Models\Faq;
use Netzperfekt\SaasFaq\SaasFaq;
use RalphJSmit\Filament\MediaLibrary\Forms\Components\MediaPicker;
use RalphJSmit\Filament\MediaLibrary\Tables\Columns\MediaColumn;

class FaqResource extends Resource
{
    protected static ?string $model = Faq::class;

    protected static ?string $navigationIcon = 'heroicon-o-question-mark-circle';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = 60;

    public static function getNavigationGroup(): ?string
    {
        return __('faq-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->count();
    }

    public static function getModelLabel(): string
    {
        return __('faq');
    }

    public static function getPluralModelLabel(): string
    {
        return __('faq');
    }

    public static function form(Form $form): Form
    {
        $faqModuleDefinition = getFaqModules();
        $thisPackage = SaasFaq::get();

        return $form
            ->columns(3)
            ->schema([
                Section::make('')
                    ->columnSpan(2)
                    ->schema([
                        TextInput::make('title')
                            ->label(__('faq-title'))
                            ->required()
                            ->maxLength(255)
                            ->autofocus()
                            ->live(debounce: 1000)
                            ->afterStateUpdated(function ($get, $set, ?string $state)
                            {
                                if(empty($get('slug'))) {
                                    $set('slug', Str::slug($state));
                                }
                            }),

                        RichEditor::make('description')
                            ->defaultRichEditor()
                            ->label(__('faq-description'))
                            ->columns(3)
                            ->required(),
                    ]),

                Section::make('')
                    ->columnSpan(1)
                    ->schema([
                        Toggle::make('is_active')
                            ->label(__('faq-active'))
                            ->default(true)
                            ->offColor('danger')
                            ->onColor('success'),

                        TextInput::make('slug')
                            ->label(__('faq-slug'))
                            ->required()
                            ->maxLength(255)
                            ->columns(1),

                        Select::make('module')
                            ->label(__('faq-module'))

                            ->required()
                            ->selectablePlaceholder(fn() => ! $thisPackage->getForTenants())
                            ->default(function() use ($faqModuleDefinition, $thisPackage)
                            {
                                if($thisPackage->getForTenants())
                                {
                                    return $faqModuleDefinition->tenantModules[$thisPackage->getForTenantsPackage()];
                                }

                                return null;
                            })
                            ->options(function() use ($faqModuleDefinition, $thisPackage)
                            {
                                if($thisPackage->getForTenants())
                                {
                                    $tenantModule = $faqModuleDefinition->tenantModules[$thisPackage->getForTenantsPackage()];
                                    return [
                                        $tenantModule => __('faq-module-' . $tenantModule)
                                    ];
                                }
                                else
                                {
                                    return $faqModuleDefinition->modules;
                                }
                            })
                            ->columns(1),

                        MediaPicker::make('image_id')
                            ->label(__('faq-image'))
                            ->columns(1)
                            ->maxWidth(MaxWidth::ExtraSmall),
                    ]),

                ...self::getFaqBuilder()
            ]);
    }

    public static function table(Table $table): Table
    {
        $faqModuleDefinition = getFaqModules();
        $thisPackage = SaasFaq::get();

        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('faq-title'))
                    ->weight('font-semibold')
                    ->searchable()
                    ->sortable(),

                MediaColumn::make('thumbnail')
                    ->label(__('faq-thumbnail')),

                Tables\Columns\TextColumn::make('slug')
                    ->label(__('faq-slug'))
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('faq_groups_count')
                    ->counts('faqGroups')
                    ->label(__('faq-groups-count'))
                    ->badge()
                    ->sortable(),

                Tables\Columns\TextColumn::make('module')
                    ->label(__('faq-module'))
                    ->badge()
                    ->formatStateUsing(function (string $state) use ($faqModuleDefinition, $thisPackage)
                    {
                        if($thisPackage->getForTenants())
                        {
                            return __('faq-module-' . $faqModuleDefinition->tenantModules[$thisPackage->getForTenantsPackage()])
                                    ?? '(' . $state . ')';
                        }
                        else
                        {
                            return $faqModuleDefinition->modules[$state] ?? '(' . $state . ')';
                        }
                    }),

                Tables\Columns\ToggleColumn::make('is_active')
                    ->label(__('faq-active'))
                    ->offColor('danger')
                    ->onColor('success')
            ])

            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('module')
                    ->label(__('faq-module'))
                    ->options($faqModuleDefinition->modules),

                Tables\Filters\SelectFilter::make('module')
                    ->label(__('faq-groups'))
                    ->relationship('faqGroups', 'title')
            ])

            ->headerActions([
                ExportAction::make()
                    ->label('Export')
                    ->exporter(FaqExporter::class),

                ImportAction::make()
                    ->label('Import')
                    ->importer(FaqImporter::class)
            ])

            ->actions([
                Tables\Actions\EditAction::make()->label('')->tooltip('Bearbeiten')->iconSize(IconSize::Medium),
                
                Tables\Actions\ReplicateAction::make()
                    ->Label('')
                    ->tooltip('Duplizieren')
                    ->iconSize(IconSize::Medium)
                    ->excludeAttributes(['faq_groups_count']),

                Tables\Actions\DeleteAction::make()->label('')->tooltip('Löschen')->iconSize(IconSize::Medium),
                Tables\Actions\ForceDeleteAction::make()->label('')->tooltip('Endgültig löschen')->iconSize(IconSize::Medium),
                Tables\Actions\RestoreAction::make()->label('')->tooltip('Wiederherstellen')->iconSize(IconSize::Medium)
            ])

            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\ExportBulkAction::make()->exporter(FaqExporter::class),

                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])

            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListFaqPages::route('/'),
            'create' => CreateFaqPage::route('/create'),
            'edit'   => EditFaqPage::route('/{record}/edit')
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }

    private static function getFaqBuilder(): array
    {
        return [
            Builder::make('contents')
                ->htmlLabel(__('faq-contents'), 'font-bold text-success text-2xl')
                ->addActionLabel(__('add-new-faq-section'))
                ->deleteAction(fn (\Filament\Forms\Components\Actions\Action $action) => $action->requiresConfirmation())
                ->cloneable()
                ->collapsible()
                ->blockNumbers(false)
                ->columnSpanFull()

                ->blocks([
                    Builder\Block::make('section')
                        ->label(function($state)
                        {
                            return new HtmlString('<b class="text-primary text-xl">' . ($state['title'] ?? __('faq-section')) . '</b>');
                        })
                        ->schema([
                            TextInput::make('title')
                                ->label(__('faq-section-title'))
                                ->required(),

                            Builder::make('section-contents')
                                ->label(__('faq-items'))
                                ->addActionLabel(__('add-new-faq-item'))
                                ->deleteAction(fn (\Filament\Forms\Components\Actions\Action $action) => $action->requiresConfirmation())
                                ->cloneable()
                                ->blockNumbers(false)
                                ->collapsible()
                                ->columnSpanFull()
                                ->blocks([
                                    Builder\Block::make('items')
                                        ->label(function($state)
                                        {
                                            return new HtmlString('<b class="text-info text-lg">' . ($state['title'] ?? __('faq-item')) . '</b>');
                                        })
                                        ->columns(7)
                                        ->schema([
                                            TextInput::make('title')
                                                ->label(__('faq-item-title'))
                                                ->columnSpan(2)
                                                ->required(),

                                            Placeholder::make('')
                                                ->columnSpan(2),

                                            TextInput::make('icon')
                                                ->label(__('faq-icon'))
                                                ->columns(1),

                                            TextInput::make('id')
                                                ->hidden()
                                                ->default(Str::uuid()->toString()),

                                            TextInput::make('counter')
                                                ->label(__('faq-counter'))
                                                ->numeric()
                                                ->minValue(0)
                                                ->default(0)
                                                ->columns(1),

                                            Toggle::make('sticky')
                                                ->label(__('faq-sticky'))
                                                ->columns(1),

                                            RichEditor::make('description')
                                                ->defaultRichEditor()
                                                ->label(__('faq-item-contents'))
                                                ->required()
                                                ->columnSpanFull(),
                                        ])
                                ])
                        ])
                        ->columns(3)
                ])
        ];
    }
}
