<?php

namespace Netzperfekt\SaasProptrend\Filament\Pages;

use Awcodes\TableRepeater\Components\TableRepeater;
use Awcodes\TableRepeater\Header;
use Filament\Actions\Action;
use Filament\Facades\Filament;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasProptrend\Data\PropertyData;
use Netzperfekt\SaasProptrend\Enums\FetchStatus;
use Netzperfekt\SaasProptrend\Jobs\ProcessPropertySave;
use Netzperfekt\SaasProptrend\Models\ProptrendProjects;
use Netzperfekt\SaasProptrend\Models\ProptrendSource;

class PropEntry extends Page
{
    protected static ?string $navigationIcon = 'fas-square-plus';
    protected static string $view = 'saas-proptrend::filament.pages.propentry';
    protected static ?int $navigationSort = -99;

    public static function getNavigationGroup(): ?string
    {
        return config('app.name');
    }

    public ?array $data = [];
    public ?array $fields = [];
    public ?string $iframe = '';

    protected function getActions(): array
    {
        return [];
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')->keyBindings(['command+s', 'ctrl+s'])
                ->label(__('save'))
                ->submit('data')
        ];
    }

    protected function getFormStatePath(): string
    {
        return 'data';
    }

    protected function getFormSchema(): array
    {
        // TODO js dom picker
        // https://github.com/hmarr/pick-dom-element

        return [
            Select::make('project_id')
                ->label(__('proptrend-project'))
                ->required()
                ->options(ProptrendProjects::onlyMyRecords()->withoutTrashed()->orderBy('title')->pluck('title', 'id'))
                ->searchable()
                ->preload()
                ->live()
                ->afterStateUpdated(function ($state, $set)
                {
                    $fields = ProptrendProjects::find($state)->fields;
                    foreach($fields as &$field)
                    {
                        $field['path'] = '';
                        $field['value'] = '';
                    }

                    $set('fields', $fields);
                }),

            TextInput::make('url')
                ->label(__('proptrend-url'))
                ->live(true)
                ->disabled(fn($get) => ! $get('project_id'))
                ->suffixAction(
                    \Filament\Forms\Components\Actions\Action::make('fetchurl')
                        ->label(__('proptrend-fetch-url'))
                        ->icon('fas-download')
                        ->requiresConfirmation()
                        ->action(function ($get, $set, $livewire)
                        {
                            $url = $get('url');

                            $information = fetchUrl($url);
                            if($information['_status'] == FetchStatus::OK)
                            {
                                $this->iframe = $information['_html'] ?? '';
                            }
                            else
                            {
                                $this->iframe = $information['_error'] ?? '';
                            }

                            if(isset($information['place']))
                            {
                                $set('place', $information['place']);
                            }

                            $sourceId = $this->getSourceFrom($get('project_id'), $url);
                            if($sourceId)
                            {
                                $set('source_id', $sourceId);
                            }
                        })
                ),

            Select::make('source_id')
                ->label(__('proptrend-source'))
                ->required()
                ->options(ProptrendSource::onlyMyRecords()->withoutTrashed()->orderBy('title')->pluck('title', 'id'))
                ->searchable()
                ->preload()
                ->live(),

            TextInput::make('place')
                ->label(__('proptrend-place'))
                ->disabled(fn($get) => ! $get('project_id'))
                ->required(),

            TableRepeater::make('fields')
                ->label(__('proptrend-fields'))
                ->disabled(fn($get) => ! $get('project_id'))
                ->addable(false)
                ->deletable(false)
                ->reorderable(false)
                ->headers([
                    Header::make('label')
                        ->label(__('proptrend-field-label'))
                        ->width('250px'),
                    Header::make('path')
                        ->label(__('proptrend-field-path'))
                        ->width('250px'),
                    Header::make('value')
                        ->label(__('proptrend-field-value')),
                ])
                ->schema([
                    TextInput::make('id')->hidden(),
                    Select::make('type')->hidden(),
                    TextInput::make('label')->disabled()->dehydrated(),

                    TextInput::make('path')
                        ->extraAttributes(fn($get) => ['id' => 'field-path-' . $get('id')]),

                    TextInput::make('value')
                        ->extraAttributes(fn($get) => ['id' => 'field-value-' . $get('id')])
                        ->suffixAction(
                            \Filament\Forms\Components\Actions\Action::make('pickvalue')
                                ->label(__('proptrend-pick-value'))
                                ->icon('fas-pencil')
                                ->action(function ($livewire, $get)
                                {
                                    $livewire->js("pickValue('#external-webcontent', '" .  $get('id') . "', '" . $get('label') . "', '" . $get('type') . "')");
                                })
                        )
                ]),

            Placeholder::make('iframe')
                ->label(__('proptrend-url-contents'))
                ->extraAttributes(['id' => 'external-webcontent', 'class' => 'border-2 border-blue-400'])
                ->content(function()
                {
                    return new HtmlString($this->iframe);
                })
        ];
    }

    public function mount(): void
    {
        $this->form->fill();
    }

    public function submit(): void
    {
        $data = $this->form->getState();
        $propertyData = PropertyData::from([
            'tenantId'  => Filament::getTenant()->id,
            'projectId' => $data['project_id'],
            'sourceId'  => $data['source_id'],
            'url'       => $data['url'],
            'place'     => $data['place'],
            'fields'    => $data['fields'],
            'html'      => $this->iframe
        ]);

        if($this->saveProp($propertyData, Auth::user()))
        {
            Notification::make()
                ->title(__('msg.prop_saving'))
                ->success()
                ->send();
        }
        else
        {
            Notification::make()
                ->title(__('msg.prop_already_exists'))
                ->danger()
                ->send();
        }
    }

    public static function getNavigationLabel(): string
    {
        return __('proptrend-propentry-label');
    }

    public function getTitle(): string
    {
        return __('proptrend-propentry-label');
    }

    private function getSourceFrom(int $projectId, string $url): ?int
    {
        $project = ProptrendProjects::with('sources')->find($projectId);
        if( ! $project)
        {
            return null;
        }

        $url = trim($url, ' /');

        $sourceId = null;
        foreach($project->sources as $source)
        {
            if(str_starts_with($url, $source->url))
            {
                $sourceId = $source->id;
            }
        }

        if( ! $sourceId)
        {
            $urlParsed = parse_url($url);

            $newSource = ProptrendSource::create([
                'project_id' => $projectId,
                'url'        => $urlParsed['scheme'] . '://' . $urlParsed['host'],
                'title'      => $urlParsed['host'],
                'field_mappings' => []
            ]);
            $sourceId = $newSource->id;
        }

        return $sourceId;
    }

    private function saveProp(PropertyData $propertyData, Authenticatable $user): bool
    {
        $projectProps = ProptrendProjects::find($propertyData->projectId)
            ->props
            ->where('url', '=', $propertyData->url);

        if($projectProps->count() > 0)
        {
            return false;
        }

        ProcessPropertySave::dispatch($propertyData, $user);

        return true;
    }
}
