<?php

namespace Netzperfekt\SaasProptrend\Filament\Resources\ProjectResource;

use Awcodes\TableRepeater\Components\TableRepeater;
use Awcodes\TableRepeater\Header;
use Filament\Forms\Components\KeyValue;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Netzperfekt\SaasProptrend\Enums\FieldType;
use Netzperfekt\SaasProptrend\Enums\ProjectStatus;
use Netzperfekt\SaasProptrend\Enums\ProjectType;
use Netzperfekt\SaasProptrend\Filament\Resources\ProjectResource\Pages\CreateProjectPage;
use Netzperfekt\SaasProptrend\Filament\Resources\ProjectResource\Pages\EditProjectPage;
use Netzperfekt\SaasProptrend\Filament\Resources\ProjectResource\Pages\ListProjectPages;
use Netzperfekt\SaasProptrend\Models\ProptrendProjects;

class ProjectResource extends Resource
{
    protected static ?string $model = ProptrendProjects::class;

    protected static ?string $navigationIcon = 'fas-binoculars';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = -2;

    public static function getNavigationGroup(): ?string
    {
        return __('proptrend-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->withoutTrashed()->count();
    }

    public static function getModelLabel(): string
    {
        return __('proptrend-project');
    }

    public static function getPluralModelLabel(): string
    {
        return __('proptrend-projects');
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['title'];
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Typ' => $record->type->getLabel()
        ];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Split::make([
                    Section::make('')
                        ->compact()
                        ->schema([
                            TextInput::make('title')
                                ->label(__('proptrend-title'))
                                ->extraAttributes(['class' => 'font-semibold'])
                                ->required()
                                ->maxLength(255)
                                ->autofocus(),

                            TableRepeater::make('fields')
                                ->label(__('proptrend-fields'))
                                ->headers([
                                    Header::make('id')
                                        ->label(__('proptrend-field-id'))
                                        ->width('100px'),
                                    Header::make('label')
                                        ->label(__('proptrend-field-label'))
                                        ->width('250px'),
                                    Header::make('type')
                                        ->label(__('proptrend-field-type'))
                                        ->width('150px'),
                                ])
                                ->schema([
                                    TextInput::make('id')
                                        ->required()
                                        ->disabled(fn($state) => $state != '')
                                        ->dehydrated(),

                                    TextInput::make('label')
                                        ->required(),

                                    Select::make('type')
                                          ->options(FieldType::class)
                                          ->required()
                                ])
                                ->columnSpan('full')

                        ])->compact()
                          ->grow(),

                    Section::make([
                        Select::make('type')
                            ->label(__('proptrend-type'))
                            ->required()
                            ->options(ProjectType::class)
                            ->live()
                            ->afterStateUpdated(function ($get, $set, $operation)
                            {
                                if($operation === 'create')
                                {
                                    $set('fields', ProjectType::getDefaultFieldsFor($get('type')));
                                }
                            }),

                        Select::make('status')
                            ->required()
                            ->label(__('proptrend-status'))
                            ->options(ProjectStatus::class)
                            ->default(ProjectStatus::Active)

                    ])->grow(false)->live()
                ])->from('xl')
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultPaginationPageOption(25)

            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('proptrend-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('type')
                    ->label(__('proptrend-type'))
                    ->badge()
                    ->color(Color::Green),

                Tables\Columns\TextColumn::make('status')
                    ->label(__('proptrend-status'))
                    ->badge()
                    ->color(Color::Blue)
            ])
            
            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('type')
                    ->label(__('proptrend-type'))
                    ->options(ProjectType::class)
            ])

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),
                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])
            
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListProjectPages::route('/'),
            'create' => CreateProjectPage::route('/create'),
            'edit' => EditProjectPage::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
