<?php

namespace Netzperfekt\SaasProptrend\Filament\Resources\PropResource;

use Carbon\Carbon;
use Cheesegrits\FilamentGoogleMaps\Fields\Map;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconSize;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use IbrahimBougaoua\FilamentRatingStar\Forms\Components\RatingStar;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Collection;
use Illuminate\Support\HtmlString;
use Netzperfekt\SaasProptrend\Enums\FieldType;
use Netzperfekt\SaasProptrend\Enums\PropStatus;
use Netzperfekt\SaasProptrend\Filament\Resources\PropResource\Pages\CreatePropPage;
use Netzperfekt\SaasProptrend\Filament\Resources\PropResource\Pages\EditPropPage;
use Netzperfekt\SaasProptrend\Filament\Resources\PropResource\Pages\ListPropPages;
use Netzperfekt\SaasProptrend\Filament\Resources\PropResource\RelationManagers\HistoryRelationManager;
use Netzperfekt\SaasProptrend\Models\ProptrendProp;

class PropResource extends Resource
{
    protected static ?string $model = ProptrendProp::class;

    protected static ?string $navigationIcon = 'fas-house';
    protected static ?string $recordTitleAttribute = 'place';
    protected static ?int $navigationSort = -2;

    public static function getNavigationGroup(): ?string
    {
        return __('proptrend-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->withoutTrashed()->count();
    }

    public static function getModelLabel(): string
    {
        return __('proptrend-prop');
    }

    public static function getPluralModelLabel(): string
    {
        return __('proptrend-props');
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['place'];
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Projekt' => $record->project->title,
            'Quelle' => $record->source->title
        ];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->disabled(fn($record) => $record?->removed_at !== null ?? false)
            ->schema([
                Split::make([
                    Section::make(__('proptrend-maindata'))
                    ->schema([
                        Section::make('')
                            ->compact()
                            ->columns(3)
                            ->schema([
                                TextInput::make('url')
                                    ->label(__('proptrend-url'))
                                    ->columnSpan(3)
                                    ->url()
                                    ->maxLength(255)
                                    ->live()
                                    ->disabled(fn($operation) => $operation != 'create')
                                    ->dehydrated()
                                    ->suffixAction(Action::make('open-event-url')
                                        ->icon('fas-link')
                                        ->iconSize(IconSize::Large)
                                        ->iconButton()
                                        ->url(fn($get) => $get('url'))
                                        ->disabled(fn($get) => $get('url') == '')
                                        ->openUrlInNewTab()),

                                Grid::make()
                                    ->columns(4)
                                    ->schema([
                                        Placeholder::make('qm-house')
                                            ->label(__('proptrend-qm-house'))
                                            ->extraAttributes(['class' => 'font-semibold text-orange-500'])
                                            ->content(fn($record) => $record?->getField(FieldType::QmHouse) . FieldType::QmHouse->getUnit() ?? ''),

                                        Placeholder::make('price')
                                            ->label(__('proptrend-price'))
                                            ->extraAttributes(['class' => 'font-semibold text-orange-500'])
                                            ->content(fn($record) => number_format((float)$record?->getField(FieldType::Price), 0, ',', '.') . FieldType::Price->getUnit() ?? 0),

                                        Placeholder::make('provision')
                                            ->label(__('proptrend-provision'))
                                            ->extraAttributes(['class' => 'font-semibold text-orange-500'])
                                            ->content(fn($record) => $record?->getField(FieldType::Provision) . FieldType::Provision->getUnit() ?? ''),

                                        Placeholder::make('total_price')
                                            ->label(__('proptrend-total-price'))
                                            ->extraAttributes(['class' => 'font-bold text-orange-500'])
                                            ->hintIcon('heroicon-m-question-mark-circle', __('proptrend-hint-total-price'))
                                            ->content(fn($get, $record)
                                                => number_format(calculateTotalPrice(
                                                    $record?->getField(FieldType::Price ?? ''),
                                                    $record?->getField(FieldType::Provision ?? 0),
                                                    $get('state') ?? ''
                                                ), 0, ',', '.') . ' €'),
                                ]),

                                Grid::make()
                                    ->columns(4)
                                    ->schema([
                                        TextInput::make('place')
                                            ->label(__('proptrend-place'))
                                            ->columnSpan(1),

                                        TextInput::make('latitude')
                                            ->label(__('proptrend-latitude'))
                                            ->columnSpan(1),

                                        TextInput::make('longitude')
                                            ->label(__('proptrend-longitude'))
                                            ->columnSpan(1),

                                        TextInput::make('state')
                                            ->label(__('proptrend-state'))
                                            ->disabled()
                                            ->dehydrated(),
                                    ]),

                                // https://filamentphp.com/plugins/cheesegrits-google-maps
                                Map::make('location')
                                   ->label(__('proptrend-location'))
                                    ->columnSpanFull()

                                    ->mapControls([
                                        'mapTypeControl'    => true,
                                        'scaleControl'      => true,
                                        'streetViewControl' => false,
                                        'rotateControl'     => false,
                                        'fullscreenControl' => true,
                                        'searchBoxControl'  => false,
                                        'zoomControl'       => true,
                                    ])
                                    ->autocomplete('place')
                                    ->autocompleteReverse(true)
                                    ->reverseGeocode([
                                        'street' => '%S %n',
                                        'city' => '%L',
                                        'state' => '%A1',
                                        'zip' => '%z',
                                        'country' => '%c'
                                    ])
                                    ->reverseGeocodeUsing(function (callable $set, array $results)
                                    {
                                        $location = $results['results'][0]['geometry']['location'];
                                        $set('latitude', $location['lat']);
                                        $set('longitude', $location['lng']);
                                    })
                                    ->placeUpdatedUsing(function (callable $set, array $place)
                                    {
                                        $stateComponent = Collection::make($place['address_components'])->search(function ($value)
                                        {
                                            return in_array('administrative_area_level_1', $value['types']);
                                        });

                                        $set('latitude', $place['geometry']['location']['lat']);
                                        $set('longitude', $place['geometry']['location']['lng']);
                                        $set('state', $place['address_components'][$stateComponent]['short_name'] ?? '');
                                    })
                                    ->defaultZoom(18)
                                    ->draggable()
                                    ->clickable(true),
                            ]),

                        Section::make('')
                            ->compact()
                            ->columns(2)
                            ->schema([
                                Textarea::make('comments_pro')
                                    ->label(__('proptrend-comments-pro'))
                                    ->columnSpan(1)
                                    ->maxLength(1024),

                                Textarea::make('comments_contra')
                                    ->label(__('proptrend-comments-contra'))
                                    ->columnSpan(1)
                                    ->maxLength(1024),

                                RatingStar::make('rating')
                                    ->label(__('proptrend-rating'))
                            ])
                    ])->compact()
                        ->collapsible()
                        ->persistCollapsed()
                        ->grow(),

                    Section::make([
                        Select::make('status')
                            ->label(__('proptrend-status'))
                            ->required()
                            ->options(PropStatus::class)
                            ->default(PropStatus::Active)
                            ->live(),

                        Select::make('project_id')
                            ->label(__('proptrend-project'))
                            ->required()
                            ->relationship(name: 'project', titleAttribute: 'title', modifyQueryUsing: fn (Builder $query) => $query->onlyMyRecords())
                            ->searchable()
                            ->preload(),

                        Select::make('source_id')
                            ->label(__('proptrend-source'))
                            ->required()
                            ->relationship(name: 'source', titleAttribute: 'title', modifyQueryUsing: fn (Builder $query) => $query->onlyMyRecords())
                            ->searchable()
                            ->preload(),

                        TagsInput::make('tags')
                            ->label(__('proptrend-tags'))
                            ->maxWidth(MaxWidth::ExtraSmall),

                        DatePicker::make('published_at')
                            ->label(__('proptrend-published'))
                            ->hintIcon('fas-clock', fn($record) => $record?->published_at->since() ?? ''),

                        /* TODO nicht so aussagekräftig, oder?
                        TextInput::make('created_at')
                            ->label(__('proptrend-created'))
                            ->formatStateUsing(function ($state) {
                                return Carbon::create($state)->format('d.m.Y H:i');
                            })
                            ->disabled(),
                        */

                        TextInput::make('updated_at')
                            ->label(__('proptrend-updated'))
                            ->formatStateUsing(function ($state) {
                                return Carbon::create($state)->format('d.m.Y H:i');
                            })
                            ->disabled()
                            ->visible(fn($get) => $get('removed_at') == null),

                        TextInput::make('removed_at')
                            ->label(__('proptrend-removed'))
                            ->formatStateUsing(function ($state) {
                                return $state ? Carbon::create($state)->format('d.m.Y H:i') : null;
                            })
                            ->hintIcon('fas-clock', fn($record)
                                => 'Veröffentlichungsdauer: ' .
                                    (int)$record->published_at->diffInDays($record->removed_at) .
                                    ' Tage'
                            )
                            ->disabled()
                            ->visible(fn($get) => $get('removed_at') !== null)

                    ])->grow(false),
                ])->from('xl')
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultPaginationPageOption(25)
            ->striped()

            ->columns([
                Tables\Columns\TextColumn::make('place')
                    ->label(__('proptrend-place'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),

                Tables\Columns\TextColumn::make('url')
                    ->label(__('proptrend-url'))
                    ->formatStateUsing(fn($state) => new HtmlString('&nbsp;'))
                    ->icon('fas-link')
                    ->url(fn($state) => $state)
                    ->openUrlInNewTab(),

                Tables\Columns\TextColumn::make('_qm_house')
                    ->label(__('proptrend-qm-house'))
                    ->getStateUsing(fn ($record) => $record->getField(FieldType::QmHouse))
                    ->formatStateUsing(fn ($state) => $state . FieldType::QmHouse->getUnit()),

                Tables\Columns\TextColumn::make('_price')
                    ->label(__('proptrend-price'))
                    ->getStateUsing(fn ($record) => $record->getField(FieldType::Price))
                    ->formatStateUsing(fn ($state) => number_format((float)$state, 0, ',', '.') . FieldType::Price->getUnit()),

                Tables\Columns\TextColumn::make('status')
                    ->label(__('proptrend-status'))
                    ->badge(),

                Tables\Columns\TextColumn::make('published_at')
                    ->label(__('proptrend-published'))
                    ->since(),

                Tables\Columns\TextColumn::make('project.title')
                    ->label(__('proptrend-project'))
                    ->badge()
                    ->color(Color::Green),

                \IbrahimBougaoua\FilamentRatingStar\Columns\Components\RatingStar::make('rating')
                    ->label(__('proptrend-rating')),

                Tables\Columns\TextColumn::make('source.title')
                    ->label(__('proptrend-source'))
                    ->badge(),

                Tables\Columns\TextColumn::make('updated_at')
                    ->label(__('proptrend-updated'))
                    ->since(),

                Tables\Columns\TextColumn::make('history_count')
                    ->label(__('proptrend-history'))
                    ->counts('history')
                    ->badge()
                    ->color(Color::Orange)
            ])
            
            ->filters([
                Tables\Filters\TrashedFilter::make(),

                Tables\Filters\SelectFilter::make('status')
                    ->label(__('proptrend-status'))
                    ->options(PropStatus::class),

                Tables\Filters\SelectFilter::make('project')
                    ->label(__('proptrend-project'))
                    ->relationship('project', 'title', modifyQueryUsing: fn (Builder $query) => $query->onlyMyRecords()),

                Tables\Filters\SelectFilter::make('source')
                    ->label(__('proptrend-source'))
                    ->relationship('source', 'title', modifyQueryUsing: fn (Builder $query) => $query->onlyMyRecords())
            ])

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),
                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])
            
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            HistoryRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListPropPages::route('/'),
            'create' => CreatePropPage::route('/create'),
            'edit' => EditPropPage::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
