<?php

namespace Netzperfekt\SaasProptrend\Filament\Resources\SourceResource;

use Awcodes\TableRepeater\Components\TableRepeater;
use Awcodes\TableRepeater\Header;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Split;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Netzperfekt\SaasProptrend\Enums\FieldType;
use Netzperfekt\SaasProptrend\Filament\Resources\SourceResource\Pages\CreateSourcePage;
use Netzperfekt\SaasProptrend\Filament\Resources\SourceResource\Pages\EditSourcePage;
use Netzperfekt\SaasProptrend\Filament\Resources\SourceResource\Pages\ListSourcePages;
use Netzperfekt\SaasProptrend\Models\ProptrendSource;

class SourceResource extends Resource
{
    protected static ?string $model = ProptrendSource::class;

    protected static ?string $navigationIcon = 'fab-sourcetree';
    protected static ?string $recordTitleAttribute = 'title';
    protected static ?int $navigationSort = -2;

    public static function getNavigationGroup(): ?string
    {
        return __('proptrend-navigation-group');
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::onlyMyRecords()->withoutTrashed()->count();
    }

    public static function getModelLabel(): string
    {
        return __('proptrend-source');
    }

    public static function getPluralModelLabel(): string
    {
        return __('proptrend-sources');
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['title'];
    }

    public static function getGlobalSearchResultDetails(Model $record): array
    {
        return [
            'Typ' => $record->type->getLabel()
        ];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->columns(1)
            ->schema([
                Split::make([
                    Section::make('')
                        ->compact()
                        ->columns(3)
                        ->grow()
                        ->schema([
                            Select::make('project_id')
                                ->label(__('proptrend-project'))
                                ->required()
                                ->relationship(name: 'project', titleAttribute: 'title')
                                ->searchable()
                                ->preload(),

                            TextInput::make('title')
                                ->label(__('proptrend-title'))
                                ->extraAttributes(['class' => 'font-semibold'])
                                ->required()
                                ->maxLength(255)
                                ->autofocus(),

                            TextInput::make('url')
                                ->label(__('proptrend-url'))
                                ->url()
                                ->maxLength(255)
                                ->live()
                                ->suffixAction(Action::make('open-event-url')
                                    ->icon('fas-link')
                                    ->url(fn($get) => $get('url'))
                                    ->disabled(fn($get) => $get('url') == '')
                                    ->openUrlInNewTab()),

                            TableRepeater::make('field_mappings')
                                ->label(__('proptrend-field-mappings'))
                                ->headers([
                                    Header::make('id')
                                        ->label(__('proptrend-field-id'))
                                        ->width('100px'),
                                    Header::make('path')
                                        ->label(__('proptrend-field-path'))
                                        ->width('250px')
                                ])
                                ->schema([
                                    TextInput::make('id')
                                        ->required()
                                        ->disabled(fn($state) => $state != '')
                                        ->dehydrated(),

                                    TextInput::make('path')
                                ])
                                ->columnSpan('full')

                        ])
                ])->from('xl')
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultPaginationPageOption(25)

            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label(__('proptrend-title'))
                    ->weight(FontWeight::SemiBold)
                    ->searchable(),
            ])
            
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])

            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->tooltip('Bearbeiten'),
                Tables\Actions\DeleteAction::make()->iconButton()->tooltip('Löschen'),
                Tables\Actions\ForceDeleteAction::make()->iconButton()->tooltip('Endgültig löschen'),
                Tables\Actions\RestoreAction::make()->iconButton()->tooltip('Wiederherstellen')
            ])
            
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListSourcePages::route('/'),
            'create' => CreateSourcePage::route('/create'),
            'edit' => EditSourcePage::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
